"use strict";
/**********************************************************************/
/*  Bitsight VRM Archer Integration - Update Archer TPP               */
/**********************************************************************/

/*
Purpose: 
	To update Archer with Monitored/Managed VRM Vendors with meta data.

High Level Overview:
    1. Login to Archer API to obtain session token and Archer version
    2. Obtain necessary Archer backend application, field, and values list id data to construct queries and perform updates due to uniqueness between Archer instances
    3. Create Archer Search XML criteria to identify Third Party Profiles in scope to update with Bitsight VRM data
    4. Get Bitsight list values, get Archer list values, and update Archer list values if necessary
	5. Get all Bitsight VRM vendors and data via multiple API calls
	6. Iterate and match between systems and build postbody for Archer
	7. Update Archer Third Party Profile records

 */

/********************************************************/
/* VERSIONING                                           */
/********************************************************/
/*  
	1/23/2025 - Version 1.0
    Initial Version - 
*/

/********************************************************/
/* LIBRARIES
/********************************************************/
const axios = require("axios");
const fs = require("fs"); //Filesystem
//const xpath = require('xpath');
const xmldom = require("@xmldom/xmldom");
const xml2js = require("xml2js");
var { params, ArcherTPPFieldParams } = require("./config.js");

/********************************************************/
/* MISC SETTINGS                                        */
/********************************************************/
//Verbose logging will log the post body data and create output files which takes up more disk space
var bVerboseLogging = true;

/********************************************************/
/* GENERAL VARIABLES                                    */
/********************************************************/
//General varibles which should not be changed
var bOverallSuccessful = true; //Optimistic
var sArcherSessionToken = null;

var aArcherTPPReport = []; //Stores the report of records obtained with a Bitsight VRM Domain AND without a Bitsight VRM GUID.
var ArcherValuesLists = []; //This will be used for iterating through to obtain values list value id from Archer
var BitsightVRMTags = []; //Stores the tag name and guid after obtaining from Bitsight VRM API
var BitsightVRMLifecycles = []; //Stores the lifecycles names and guid after obtaining from Bitsight VRM API
var BitsightVRMVendors = []; //Stores the main data object of Bitsight VRM Vendor data

var sErrorLogDetails = "";
var totalErrors = 0;
var totalWarnings = 0;

var totalReportRequests = 0;
var totalReportSuccess = 0;
var totalReportErrors = 0;

//Used to store files in logs subdirectory
var appPrefix = "110";

//Bitsight Tracking Stats
var COST_ArcherBitsightVersion = "RSA Archer 1.0";
var COST_Platform = "RSA Archer";
var BitsightCustomerName = "unknown";

//Just a simple way to remember the Archer input data types
var ArcherInputTypes = {
	"text": 1,
	"numeric": 2,
	"date": 3,
	"valueslist": 4,
	"externallinks": 7,
	"usergrouprecperm": 8,
	"crossref": 9,
	"attachment": 11,
	"image": 12,
	"matrix": 16,
	"ipaddress": 19,
	"relatedrecords": 23,
	"subform": 24,
};

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//BEGIN HELPER FUNCTIONS
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
function LogInfo(text) {
	//Log the details
	console.log(getDateTime() + "::INFO:: " + text);
}

function LogWarn(text) {
	//Log the details
	console.log(getDateTime() + "::WARN:: " + text);

	sErrorLogDetails += getDateTime() + "::WARN:: " + text + "\r\n";
	totalWarnings++;
}

function LogError(text) {
	//Update stat
	totalErrors++;

	//Log overall error to file
	LogSaaSError();

	//Log the details to output log file
	console.log(getDateTime() + "::ERROR:: " + text);

	//Log the error that gets sent to the API monitoring
	sErrorLogDetails += getDateTime() + "::ERROR:: " + text + "\r\n";
}

//Only log this if verbose logging is turned on - not needed unless troubleshooting
function LogVerbose(text) {
	if (bVerboseLogging) {
		console.log(getDateTime() + "::VERB:: " + text);
	}
}

//Simple function to get date and time in standard format
function getDateTime() {
	var dt = new Date();
	return (
		pad(dt.getFullYear(), 4) +
		"-" +
		pad(dt.getMonth() + 1, 2) +
		"-" +
		pad(dt.getDate(), 2) +
		" " +
		pad(dt.getHours(), 2) +
		":" +
		pad(dt.getMinutes(), 2) +
		":" +
		pad(dt.getSeconds(), 2)
	);
}

//Pads a certain amount of characters based on the size of text provided
function pad(num, size) {
	var s = num + "";
	//prepend a "0" until desired size reached
	while (s.length < size) {
		s = "0" + s;
	}
	return s;
}

//Simple method to log an error to a file for batch execution. The batch file will check if this file exists.
function LogSaaSError() {
	if (bOverallSuccessful == true) {
		bOverallSuccessful = false; //Set the flag to false so we only create this file one time and avoid file lock issues.

		fs.writeFileSync("logs\\error-" + appPrefix + ".txt", "ERROR");
		LogInfo("Logged error and created logs\\error-" + appPrefix + ".txt file.");
	}
}

//Simple method to log successful execution for execution. The batch file will check if this file exists.
function LogSaaSSuccess() {
	if (bOverallSuccessful == true) {
		fs.writeFileSync("logs\\success-" + appPrefix + ".txt", "SUCCESS");
		LogInfo("Logged success and created logs\\success-" + appPrefix + ".txt file.");
	}
}

function xmlStringToXmlDoc(xml) {
	var p = new xmldom.DOMParser();
	return p.parseFromString(xml, "text/xml");
	//return p.parseFromString(xml, "application/xml");
}

function b64Encode(str) {
	var CHARS = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
	var out = "",
		i = 0,
		len = str.length,
		c1,
		c2,
		c3;

	while (i < len) {
		c1 = str.charCodeAt(i++) & 0xff;
		if (i == len) {
			out += CHARS.charAt(c1 >> 2);
			out += CHARS.charAt((c1 & 0x3) << 4);
			out += "==";
			break;
		}

		c2 = str.charCodeAt(i++);
		if (i == len) {
			out += CHARS.charAt(c1 >> 2);
			out += CHARS.charAt(((c1 & 0x3) << 4) | ((c2 & 0xf0) >> 4));
			out += CHARS.charAt((c2 & 0xf) << 2);
			out += "=";
			break;
		}

		c3 = str.charCodeAt(i++);
		out += CHARS.charAt(c1 >> 2);
		out += CHARS.charAt(((c1 & 0x3) << 4) | ((c2 & 0xf0) >> 4));
		out += CHARS.charAt(((c2 & 0xf) << 2) | ((c3 & 0xc0) >> 6));
		out += CHARS.charAt(c3 & 0x3f);
	}
	return out;
}

function makeBasicAuth(token) {
	//Purpose of this is to convert the token to the authorization header for basic auth
	//Format is Token with a colon at the end then converted to Base64
	return b64Encode(token + ":");
}

//GetArcherText validates then returns data from a text value
function GetArcherText(sText) {
	try {
		if (typeof sText == "undefined" || typeof sText._ == "undefined" || sText == null) {
			return "";
		} else {
			return sText._.trim();
		}
	} catch (ex) {
		LogWarn("GetArcherText() Error getting Archer text. ex:" + ex);
		LogVerbose("GetArcherText() Error getting Archer text. ex.stack:" + ex.stack);
		return "";
	}
}

//GetArcherValue validates then returns data from a single value list
function GetArcherValue(jValueNode) {
	try {
		if (
			typeof jValueNode == "undefined" ||
			jValueNode == null ||
			typeof jValueNode.ListValues == "undefined" ||
			typeof jValueNode.ListValues[0] == "undefined" ||
			typeof jValueNode.ListValues[0].ListValue == "undefined" ||
			typeof jValueNode.ListValues[0].ListValue[0] == "undefined" ||
			typeof jValueNode.ListValues[0].ListValue[0]._ == "undefined"
		) {
			return "";
		} else {
			return jValueNode.ListValues[0].ListValue[0]._;
		}
	} catch (ex) {
		LogWarn("GetArcherValue() Error getting Archer single value. ex:" + ex);
		LogVerbose("GetArcherValue() Error getting Archer single value. ex.stack:" + ex.stack);
		return "";
	}
}

//GetArcherValue validates then returns data from a multi-select value list
function GetArcherValues(jValueNode) {
	LogVerbose("GetArcherValues() Start. jValueNode=" + JSON.stringify(jValueNode));

	let tmp = [];
	try {
		if (
			typeof jValueNode == "undefined" ||
			jValueNode == null ||
			typeof jValueNode.ListValues == "undefined" ||
			typeof jValueNode.ListValues[0] == "undefined" ||
			typeof jValueNode.ListValues[0].ListValue == "undefined" ||
			typeof jValueNode.ListValues[0].ListValue.length == 0 ||
			typeof jValueNode.ListValues[0].ListValue[0] == "undefined" ||
			typeof jValueNode.ListValues[0].ListValue[0]._ == "undefined"
		) {
			return [];
		} else {
			//Iterate through the nodes and return an array of values.

			for (let v in jValueNode.ListValues[0].ListValue) {
				tmp[tmp.length] = jValueNode.ListValues[0].ListValue[v]._;
				LogVerbose("GetArcherValues() v=" + v + " Value=" + jValueNode.ListValues[0].ListValue[v]._);
			}
		}
	} catch (ex) {
		LogWarn("GetArcherValues() Error iterating Archer multi-select values list. ex:" + ex);
		LogVerbose("GetArcherValues() Error iterating Archer multi-select values list. ex.stack:" + ex.stack);
	} finally {
		//Doing this in a "finally" block in case there was an error iterating and we had some data.
		LogInfo("GetArcherValues() tmp=" + JSON.stringify(tmp));
		return tmp;
	}
}

function getDateYYYYMMMDD() {
	//Returns current date in YYYY-MM-DD format which is what Archer is expecting
	try {
		var dt = new Date();
		return pad(dt.getFullYear(), 4) + "-" + pad(dt.getMonth() + 1, 2) + "-" + pad(dt.getDate(), 2);
	} catch (ex) {
		LogWarn("getDateYYYYMMMDD() - Issue creating YYYYMMDD date string. Returning emptystring. ex:" + ex.stack);
		return "";
	}
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//END HELPER FUNCTIONS
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////BEGIN CORE FUNCTIONALITY
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

async function ArcherLogin() {
	LogInfo("ArcherLogin() Start.");

	//Optimistic
	let bSuccess = true;
	let data = null;

	try {
		//construct body
		const body = {
			"InstanceName": params["archer_instanceName"],
			"Username": params["archer_username"],
			"UserDomain": "",
			"Password": params["archer_password"],
		};

		const sUrl = params["archer_webroot"] + params["archer_rest_root"] + params["archer_loginpath"];

		const httpConfig = {
			method: "POST",
			url: sUrl,
			headers: {
				"Content-Type": "application/json",
				Accept: "application/json,text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8",
			},
			data: body,
		};

		LogVerbose("ArcherLogin() API call httpConfig=" + JSON.stringify(httpConfig));

		//API call to get session token
		await axios(httpConfig)
			.then(function (res) {
				data = res.data;
				LogVerbose("ArcherLogin() Axios call complete. data=" + JSON.stringify(data));
				//Expecting data to be in the format of: {"Links":[],"RequestedObject":{"SessionToken":"823B73BA88E36B8DABB113E56DDE9FB8","InstanceName":"Archer_Bitsight67","UserId":212,"ContextType":0,"UserConfig":{"TimeZoneId":"Eastern Standard Time","TimeZoneIdSource":1,"LocaleId":"en-US","LocaleIdSource":2,"LanguageId":1,"DefaultHomeDashboardId":-1,"DefaultHomeWorkspaceId":-1,"LanguageIdSource":1,"PlatformLanguageId":1,"PlatformLanguagePath":"en-US","PlatformLanguageIdSource":1},"Translate":false,"IsAuthenticatedViaRestApi":true},"IsSuccessful":true,"ValidationMessages":[]}
			})
			.catch(function (error) {
				bSuccess = false;
				LogError("ArcherLogin() Axios call error. Err: " + error);
				LogVerbose("ArcherLogin() Axios call error. Err.stack: " + error.stack);
			});

		if (bSuccess === true) {
			//Attempt to get the session token
			if (
				typeof data != "undefined" &&
				data != null &&
				typeof data.RequestedObject != "undefined" &&
				data.RequestedObject != null &&
				typeof data.RequestedObject.SessionToken != "undefined" &&
				data.RequestedObject.SessionToken != null &&
				data.RequestedObject.SessionToken.length > 10
			) {
				sArcherSessionToken = data.RequestedObject.SessionToken;
				LogVerbose("ArcherLogin() SessionToken=" + sArcherSessionToken);
			} else {
				LogError("ArcherLogin() Archer Session Token missing.");
				bSuccess = false;
			}
		}
	} catch (ex) {
		bSuccess = false;
		LogError("ArcherLogin() Error constructing call. ex: " + ex);
		LogVerbose("ArcherLogin() Error constructing call. ex.stack: " + ex.stack);
	}

	//Return bSuccess if it passed or failed. We need the Archer session token to do anything.
	return bSuccess;
}

/********************************************************/
/********	GetArcherVersion
/********************************************************/
async function GetArcherVersion() {
	LogInfo("GetArcherVersion() Start.");
	try {
		let data = null;

		var sUrl = params["archer_webroot"] + params["archer_rest_root"] + params["archer_version"];

		const httpConfig = {
			method: "GET",
			url: sUrl,
			headers: {
				"Authorization": 'Archer session-id="' + sArcherSessionToken + '"',
				"Content-Type": "application/json",
				Accept: "application/json,text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8",
			},
			data: {},
		};

		LogVerbose("GetArcherVersion() API call httpConfig=" + JSON.stringify(httpConfig));

		//API call to get Archer Version
		await axios(httpConfig)
			.then(function (res) {
				data = res.data;
				LogVerbose("GetArcherVersion() Axios call complete. data=" + JSON.stringify(data));
				//Expecting data to be in the format of:
			})
			.catch(function (error) {
				LogWarn("GetArcherVersion() Axios call error. Err: " + error);
				LogVerbose("GetArcherVersion() Axios call error. Err.stack: " + error.stack);
			});

		//If the version is available, update the COST_Platform variable (it is generic by default...this adds the version)
		if (typeof data != "undefined" && typeof data.RequestedObject != "undefined" && data.RequestedObject.Version != "undefined") {
			let sArcherVersionNum = data.RequestedObject.Version; //Get the Version
			COST_Platform = COST_Platform + " (" + sArcherVersionNum + ")";
			LogInfo("sArcherVersionNum: " + sArcherVersionNum);
		} else {
			sArcherVersionNum = "Unknown";
			LogWarn("Unable to obtain Archer Version Number.");
		}
	} catch (ex) {
		//We won't fail on this because it's not urgent and we have default values available.
		LogWarn("GetArcherVersion() Error constructing call or parsing result. ex: " + ex);
		LogVerbose("GetArcherVersion() Error constructing call or parsing result. ex.stack: " + ex.stack);
	}
}

/********************************************************/
/********	getTPPFieldIDs
/********************************************************/
async function getArcherTPPFieldIDs() {
	LogInfo("getArcherTPPFieldIDs() Start.");

	async function getTPPModuleID() {
		LogInfo("getTPPModuleID() Start.");
		let bSuccess = true;

		try {
			let data = null;

			const sUrl = params["archer_webroot"] + params["archer_rest_root"] + params["archer_applicationpath"];
			const odataquery = "?$select=Name,Id,Guid&$filter=Name eq '" + params["archer_ThirdPartyProfileApp"] + "'";
			const postBody = {
				"Value": odataquery,
			};

			const httpConfig = {
				method: "GET",
				url: sUrl,
				headers: {
					"Authorization": 'Archer session-id="' + sArcherSessionToken + '"',
					"Content-Type": "application/json",
					Accept: "application/json,text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8",
				},
				data: postBody,
			};

			LogVerbose("getTPPModuleID() API call httpConfig=" + JSON.stringify(httpConfig));

			//API call to get Archer Third Party Profile application module ID.
			await axios(httpConfig)
				.then(function (res) {
					data = res.data;
					LogVerbose("getTPPModuleID() Axios call complete. data=" + JSON.stringify(data));
					//Expecting data to be in the format of:
				})
				.catch(function (error) {
					bSuccess = false;
					LogError("getTPPModuleID() Axios call error. Err: " + error);
					LogVerbose("getTPPModuleID() Axios call error. Err.stack: " + error.stack);
				});

			//If the module is available, use it and get the fields.
			if (typeof data != "undefined" && typeof data[0].RequestedObject != "undefined" && data[0].RequestedObject.Id != "undefined") {
				var iModuleID = data[0].RequestedObject.Id; //Get the content ID
				LogInfo("iModuleID: " + iModuleID);
				//Set as a param variable used later for search queries and record updates.
				params["archer_ThirdPartyProfileAppID"] = iModuleID;

				//Get Field IDs for the app
				bSuccess = await getFieldIDs(iModuleID); //the function will return true or false for success
			} else {
				LogError("ERROR Obtaining Third Party Profile module ID.");
				bSuccess = false;
			}
		} catch (ex) {
			LogError("getTPPModuleID() Error constructing call or parsing result. ex: " + ex);
			LogVerbose("getTPPModuleID() Error constructing call or parsing result. ex.stack: " + ex.stack);
			bSuccess = false;
		}

		return bSuccess;
	}

	async function getFieldIDs(iModuleID) {
		LogInfo("getFieldIDs() Start.");
		let bSuccess = true;

		try {
			let data = null;

			const sUrl = params["archer_webroot"] + params["archer_rest_root"] + params["archer_fielddefinitionapppath"] + iModuleID;
			//const odataquery = "?$orderby=Name&$filter=Name eq 'Bitsight Portfolio Created?' or Name eq 'Domain'"; //example
			//Ideally we could filter using an operator like "starts with" or "contains", but Archer doesn't support anything like that.
			//We could have used filters for each field name, but there are size restrictions for the odata query we would exceed.
			//So we will retrieve all fields, then parse to get the ones we care about.
			//Unfortunately Archer won't let us specify the attributes we actually need, so all are returned.
			//const odataquery = "?$select=Name,Id,Guid,LevelId,RelatedValuesListId&$orderby=Name";
			const odataquery = "?$orderby=Name";
			//Obtaining the field name, id, guid, levelId, and the relatedvalueslistId if it's a list.

			const postBody = {
				"Value": odataquery,
			};

			const httpConfig = {
				method: "GET",
				url: sUrl,
				headers: {
					"Authorization": 'Archer session-id="' + sArcherSessionToken + '"',
					"Content-Type": "application/json",
					Accept: "application/json,text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8",
				},
				data: postBody,
			};

			LogVerbose("getFieldIDs() API call httpConfig=" + JSON.stringify(httpConfig));

			//API call to get Archer Third Party Profile application module ID.
			//NOTE: Fields for ALL levels are returned, so be aware of that if we ever add fields at different levels.
			await axios(httpConfig)
				.then(function (res) {
					data = res.data;
					//LogVerbose("getFieldIDs() Axios call complete. data=" + JSON.stringify(data));

					if (bVerboseLogging === true) {
						let filename = "logs\\" + appPrefix + "\\" + "01getFieldIDs.json";
						fs.writeFileSync(filename, JSON.stringify(data));
						LogVerbose("Saved getFieldIDs data to " + filename);
					}
				})
				.catch(function (error) {
					bSuccess = false;
					LogError("getFieldIDs() Axios call error. Err: " + error);
					LogVerbose("getFieldIDs() Axios call error. Err.stack: " + error.stack);
				});

			if (typeof data != "undefined" && typeof data[0].RequestedObject != "undefined") {
				LogVerbose("getFieldIDs() Archer returned good TPP app data.");

				for (let iField in data) {
					let sFieldName = data[iField].RequestedObject.Name.toLowerCase().trim();

					//Uncomment to see all fields evaluated.
					//LogVerbose("*Looking for: " + sFieldName);

					//sField is the field name (key of the json object)
					for (let sField in ArcherTPPFieldParams) {
						//Get the value into lowercase
						let sFieldLower = sField.toLowerCase();

						//Compare the Archer value to the value we have in our ArcherTPPFieldParams object
						if (sFieldName == sFieldLower) {
							//If we have a match, then we'll get the details and set the data to the ArcherTPPFieldParams object

							let sId = data[iField].RequestedObject.Id;
							let sGuid = data[iField].RequestedObject.Guid;
							let sRelatedValuesListId = data[iField].RequestedObject.RelatedValuesListId;

							let tmp = {
								"id": sId,
								"guid": sGuid,
								"RelatedValuesListId": sRelatedValuesListId,
							};

							//Obtain Values Lists
							//If the RelatedValuesListId exists, then add it to the ArcherValuesLists array
							//We have 2 values lists that we'll need to get the values for later (Bitsight VRM Tags and Bitsight VRM Lifecycle).
							if (typeof sRelatedValuesListId != "undefined" && sRelatedValuesListId != null) {
								ArcherValuesLists[ArcherValuesLists.length] = {
									"FieldName": sField,
									"ValuesListID": sRelatedValuesListId,
									"Values": [],
								};
							}

							//Set the ArcherTPPFieldParams value for this field
							ArcherTPPFieldParams[sField] = tmp;

							//get out of this loop
							break;
						}
					}
				}

				if (bVerboseLogging === true) {
					let filename = "logs\\" + appPrefix + "\\" + "02ArcherTPPFieldParams.json";
					fs.writeFileSync(filename, JSON.stringify(ArcherTPPFieldParams));
					LogVerbose("Saved ArcherTPPFieldParams to " + filename);
				}

				//Get the values list values
				bSuccess = await getTPPValuesListValues(0);
			} else {
				bSuccess = false;
				LogError("ERROR Obtaining TPP field definitions. Cannot continue.");
			}
		} catch (ex) {
			bSuccess = false;
			LogError("getFieldIDs() Error constructing call or parsing result. ex: " + ex);
			LogVerbose("getFieldIDs() Error constructing call or parsing result. ex.stack: " + ex.stack);
		}

		return bSuccess;
	}

	async function getTPPValuesListValues(currentValuesList) {
		let bSuccess = true;
		LogInfo("----------------------------getTPPValuesListValues----------------------------");
		let data;
		try {
			LogInfo("getTPPValuesListValues() Getting VL: " + ArcherValuesLists[currentValuesList].FieldName);
			const valueslistID = ArcherValuesLists[currentValuesList].ValuesListID;
			const sUrl = params["archer_webroot"] + params["archer_rest_root"] + params["archer_valueslistvaluepath"] + valueslistID;

			const httpConfig = {
				method: "GET",
				url: sUrl,
				headers: {
					"Authorization": 'Archer session-id="' + sArcherSessionToken + '"',
					"Content-Type": "application/json",
					Accept: "application/json,text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8",
				},
				data: {},
			};

			LogInfo("getTPPValuesListValues() httpConfig: " + JSON.stringify(httpConfig));

			//API call to get values list
			await axios(httpConfig)
				.then(function (res) {
					data = res.data;
					LogVerbose("getTPPValuesListValues() Axios call complete. data=" + JSON.stringify(data));
				})
				.catch(function (error) {
					//bSuccess = false;
					//LogError("getTPPValuesListValues() Axios call error. Err: " + error);
					LogVerbose("getTPPValuesListValues() Axios call error. Err.stack: " + error.stack);
					LogWarn("getTPPValuesListValues() Axios call error. Err: " + error);
				});

			//Parse results
			if (bSuccess === true && typeof data != "undefined" && typeof data[0].RequestedObject != "undefined") {
				LogVerbose("getTPPValuesListValues() Archer returned good values list data.");

				let id = "";
				let name = "";

				for (let i in data) {
					id = data[i].RequestedObject.Id;
					name = data[i].RequestedObject.Name;

					// //Format name of value to remove spaces and /
					// name = name.replace(/\//g, "");
					// name = name.replace(/ /g, "");

					let tmp = {
						"name": name,
						"id": id,
					};

					//Set the list info to the next item in the ArcherValuesLists.Values item
					ArcherValuesLists[currentValuesList].Values[ArcherValuesLists[currentValuesList].Values.length] = tmp;
				}

				// //Iterate through if there were multiple
				// //Did we hit the max?
				// LogVerbose("getTPPValuesListValues() Current=" + currentValuesList + " Max=" + ArcherValuesLists.length);
				// if (currentValuesList >= ArcherValuesLists.length - 1) {
				// 	LogVerbose("getTPPValuesListValues() Hit maxResults of " + ArcherValuesLists.length);
				// 	if (bVerboseLogging === true) {
				// 		var fs = require("fs");
				// 		fs.writeFileSync("logs\\" + appPrefix + "\\" + "ArcherValuesLists.json", JSON.stringify(ArcherValuesLists));
				// 		LogVerbose("Saved to logs\\" + appPrefix + "\\" + "ArcherValuesLists.json file");
				// 	}
				// 	return bSuccess;
				// } else {
				// 	//Still have more values lists to iterate through...
				// 	currentValuesList++; //Increment before running again
				// 	LogVerbose("getTPPValuesListValues() Iterating through next ValuesList=" + currentValuesList);
				// 	bSuccess = await getTPPValuesListValues(currentValuesList);
				// }
			} else {
				// bSuccess = false;
				// LogSaaSError();
				// LogError("getTPPValuesListValues() ERROR Obtaining TPP field values list ids.");
				LogWarn("getTPPValuesListValues() ERROR Obtaining TPP field values list ids.");
			}

			//4/24/2025 Restructured because empty lists cause 500 error. can we move this here?
			//Iterate through if there were multiple
			//Did we hit the max?
			LogVerbose("getTPPValuesListValues() Current=" + currentValuesList + " Max=" + ArcherValuesLists.length);
			if (currentValuesList >= ArcherValuesLists.length - 1) {
				LogVerbose("getTPPValuesListValues() Hit maxResults of " + ArcherValuesLists.length);
				if (bVerboseLogging === true) {
					var fs = require("fs");
					fs.writeFileSync("logs\\" + appPrefix + "\\" + "ArcherValuesLists.json", JSON.stringify(ArcherValuesLists));
					LogVerbose("Saved to logs\\" + appPrefix + "\\" + "ArcherValuesLists.json file");
				}
				return bSuccess;
			} else {
				//Still have more values lists to iterate through...
				currentValuesList++; //Increment before running again
				LogVerbose("getTPPValuesListValues() Iterating through next ValuesList=" + currentValuesList);
				bSuccess = await getTPPValuesListValues(currentValuesList);
			}
			//END of 4/24/2025 updates
		} catch (ex) {
			bSuccess = false;
			LogError("getTPPValuesListValues() Error constructing call or parsing result. ex: " + ex);
			LogVerbose("getTPPValuesListValues() Error constructing call or parsing result. ex.stack: " + ex.stack);
		}

		return bSuccess;
	}

	//Start of all the functions to get TPP ids
	return await getTPPModuleID();
}

//Function to build search criteria and get report of TPPs needing a Bitsight GUID.
async function getArcherTPPsWithDomain() {
	LogInfo("getArcherTPPsWithDomain() Start.");
	let bSuccess = true;
	let data = null;
	try {
		//Construct search query based on data obtained an populated in the ArcherTPPFieldParams object (it has the guids for the fields)
		let sSearchCriteria =
			"<SearchReport><PageSize>10000</PageSize><MaxRecordCount>10000</MaxRecordCount>" +
			'<DisplayFields><DisplayField name="Bitsight VRM Domain">' +
			ArcherTPPFieldParams["Bitsight VRM Domain"].guid +
			'</DisplayField><DisplayField name="Bitsight GUID">' +
			ArcherTPPFieldParams["Bitsight GUID"].guid +
			'</DisplayField><DisplayField name="Bitsight Security Rating">' +
			ArcherTPPFieldParams["Bitsight Security Rating"].guid +
			'</DisplayField><DisplayField name="Bitsight VRM Deep Link">' +
			ArcherTPPFieldParams["Bitsight VRM Deep Link"].guid +
			'</DisplayField><DisplayField name="Bitsight VRM GUID">' +
			ArcherTPPFieldParams["Bitsight VRM GUID"].guid +
			'</DisplayField><DisplayField name="Bitsight VRM Impact Score">' +
			ArcherTPPFieldParams["Bitsight VRM Impact Score"].guid +
			'</DisplayField><DisplayField name="Bitsight VRM Last Updated Date">' +
			ArcherTPPFieldParams["Bitsight VRM Last Updated Date"].guid +
			'</DisplayField><DisplayField name="Bitsight VRM Lifecycle">' +
			ArcherTPPFieldParams["Bitsight VRM Lifecycle"].guid +
			'</DisplayField><DisplayField name="Bitsight VRM Requirements Completion %">' +
			ArcherTPPFieldParams["Bitsight VRM Requirements Completion %"].guid +
			'</DisplayField><DisplayField name="Bitsight VRM Requirements Due Date">' +
			ArcherTPPFieldParams["Bitsight VRM Requirements Due Date"].guid +
			'</DisplayField><DisplayField name="Bitsight VRM Review Year">' +
			ArcherTPPFieldParams["Bitsight VRM Review Year"].guid +
			'</DisplayField><DisplayField name="Bitsight VRM Risk Score">' +
			ArcherTPPFieldParams["Bitsight VRM Risk Score"].guid +
			'</DisplayField><DisplayField name="Bitsight VRM Tags">' +
			ArcherTPPFieldParams["Bitsight VRM Tags"].guid +
			'</DisplayField><DisplayField name="Bitsight VRM Trust Score">' +
			ArcherTPPFieldParams["Bitsight VRM Trust Score"].guid +
			'</DisplayField></DisplayFields><Criteria><ModuleCriteria><Module name="Third Party Profile">' +
			params["archer_ThirdPartyProfileAppID"] +
			'</Module><SortFields><SortField name="Sort1"><Field name="Bitsight VRM Domain">' +
			ArcherTPPFieldParams["Bitsight VRM Domain"].guid +
			"</Field><SortType>Ascending</SortType></SortField></SortFields></ModuleCriteria><Filter>" +
			'<Conditions><TextFilterCondition name="Text 1"><Field name="Bitsight VRM Domain">' +
			ArcherTPPFieldParams["Bitsight VRM Domain"].guid +
			"</Field><Operator>DoesNotEqual</Operator><Value></Value></TextFilterCondition></Conditions>" +
			"</Filter></Criteria></SearchReport>";

		LogVerbose("getArcherTPPsWithDomain() sSearchCriteria=" + sSearchCriteria);

		/* build url */
		var sUrl = params["archer_webroot"] + params["archer_ws_root"] + params["archer_searchpath"];

		var headers = {
			"content-type": "text/xml; charset=utf-8",
			"SOAPAction": "http://archer-tech.com/webservices/ExecuteSearch",
		};

		//Must escape the XML to next inside of the soap request...
		sSearchCriteria = sSearchCriteria.replace(/&/g, "&amp;").replace(/</g, "&lt;").replace(/>/g, "&gt;").replace(/"/g, "&quot;").replace(/'/g, "&apos;");

		const postBody =
			'<?xml version="1.0" encoding="utf-8"?><soap:Envelope xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xmlns:xsd="http://www.w3.org/2001/XMLSchema" xmlns:soap="http://schemas.xmlsoap.org/soap/envelope/">' +
			'<soap:Body><ExecuteSearch xmlns="http://archer-tech.com/webservices/"><sessionToken>' +
			sArcherSessionToken +
			"</sessionToken>" +
			"<searchOptions>" +
			sSearchCriteria +
			"</searchOptions><pageNumber>1</pageNumber></ExecuteSearch></soap:Body></soap:Envelope>";

		const httpConfig = {
			method: "POST",
			url: sUrl,
			headers: headers,
			data: postBody,
		};

		//Execute API Search Query
		LogVerbose("getArcherTPPsWithDomain() API call httpConfig=" + JSON.stringify(httpConfig));

		await axios(httpConfig)
			.then(function (res) {
				data = res.data;
				LogVerbose("getArcherTPPsWithDomain() Axios call complete. data=" + data);

				if (bVerboseLogging === true) {
					let filename = "logs\\" + appPrefix + "\\" + "03getArcherTPPsWithDomain.xml";
					fs.writeFileSync(filename, data);
					LogVerbose("Saved getArcherTPPsWithDomain data to " + filename);
				}
			})
			.catch(function (error) {
				bSuccess = false;
				LogError("getArcherTPPsWithDomain() Axios call error. Err: " + error);
				LogVerbose("getArcherTPPsWithDomain() Axios call error. Err.stack: " + error.stack);
			});

		//Need to parse the data here, but want to take a look at it first.
	} catch (ex) {
		bSuccess = false;
		LogError("getArcherTPPsWithDomain() Error constructing call or parsing result. ex: " + ex);
		LogVerbose("getArcherTPPsWithDomain() Error constructing call or parsing result. ex.stack: " + ex.stack);
	}

	if (bSuccess === true) {
		bSuccess = await parseArcherTPPRecords(data);
	}

	return bSuccess;
}

async function parseArcherTPPRecords(data) {
	LogInfo("parseArcherTPPRecords() Start.");
	let bSuccess = true;
	//variable for our json object
	let resultJSON = null;

	try {
		//Convert XML data results to an XMLDOM for parsing
		let doc = xmlStringToXmlDoc(data);

		//Check to see if nothing was returned from the search query
		if (
			typeof doc.getElementsByTagName("ExecuteSearchResult") == "undefined" ||
			typeof doc.getElementsByTagName("ExecuteSearchResult")[0] == "undefined" ||
			typeof doc.getElementsByTagName("ExecuteSearchResult")[0].childNodes == "undefined" ||
			typeof doc.getElementsByTagName("ExecuteSearchResult")[0].childNodes[0] == "undefined" ||
			typeof doc.getElementsByTagName("ExecuteSearchResult")[0].childNodes[0].nodeValue == "undefined"
		) {
			//There weren't any records
			LogInfo("parseArcherTPPRecords() No TPPs with a Bitsight VRM Domain AND without a Bitsight VRM GUID. Exiting 01.");
			//aArcherTPPReport will remain empty as a result, but technically this was successful.
			return bSuccess;
		} //Need to proceed and check the count anyway.
		else {
			//let tmp = new xmldom.XMLSerializer().serializeToString(doc);
			//console.log("----------------------------------------------------------------------------------------------------------------------");
			//console.log("doc=" + tmp);

			//Need to get the xml inside the SOAP request and url decode the results
			//ORIG:var sXML = doc.getElementsByTagName("ExecuteSearchResult")[0].childNodes[0].nodeValue;
			let sXML = doc.getElementsByTagName("ExecuteSearchResult")[0].textContent; //New method after upgrading xmldom

			//console.log("----------------------------------------------------------------------------------------------------------------------");
			//console.log("sXML=" + sXML);

			// tmp = new xmldom.XMLSerializer().serializeToString(sXML);
			// console.log("----------------------------------------------------------------------------------------------------------------------");
			// console.log("sXMLParsed=" + tmp);

			//turn the xml results into the json object
			xml2js.parseString(sXML, function (err, result) {
				resultJSON = result; //get the result into the object we can use below;
			});
			//console.log("----------------------------------------------------------------------------------------------------------------------");

			//let JSONText = JSON.stringify(resultJSON);
			console.log("resultJSON=" + JSON.stringify(resultJSON));
			//console.log("resultJSON=" + JSONText);

			if (bVerboseLogging === true) {
				let filename = "logs\\" + appPrefix + "\\" + "04archerTPPs.json";
				let fs = require("fs");
				fs.writeFileSync(filename, JSON.stringify(resultJSON));
				LogVerbose("Saved resultJSON data to " + filename);
			}

			let iNumCompanies = resultJSON.Records.$.count; //Get the number of record returned
			LogInfo("iNumCompanies=" + iNumCompanies);

			//Set overall stats
			totalReportRequests = parseInt(iNumCompanies);

			//Check to see if we have any existing records
			if (iNumCompanies == 0) {
				LogInfo("parseArcherTPPRecords() No TPPs with a Bitsight VRM Domain AND without a Bitsight VRM GUID. Exiting 02.");
				//aArcherTPPReport will remain empty as a result, but technically this was successful.
				//This will happen the majority of the time for this application.
				return bSuccess;
			}
		}
	} catch (ex) {
		bSuccess = false;
		LogError("parseArcherTPPRecords() Error parsing Archer result. ex: " + ex);
		LogVerbose("parseArcherTPPRecords() Error parsing Archer result. ex.stack: " + ex.stack);
		return bSuccess;
	}

	try {
		params["archer_ThirdPartyProfileLevelID"] = resultJSON.Records.LevelCounts[0].LevelCount[0].$.id;
	} catch (ex) {
		bSuccess = false;
		LogError("parseArcherTPPRecords() Error parsing TPP Archer level id data. Cannot continue. ex: " + ex);
		LogVerbose("parseArcherTPPRecords() Error parsing TPP Archer level id data. Cannot continue. ex.stack: " + ex.stack);
		return bSuccess;
	}

	//If we got this far, we didn't have errors AND we have data.
	try {
		let iID_TPPBitsightVRMDomain;
		let iID_TPPBitsightVRMTags;
		let iID_TPPBitsightVRMDeepLink;
		let iID_TPPBitsightSecurityRating;
		let iID_TPPBitsightVRMGUID;
		let iID_TPPBitsightVRMImpactScore;
		let iID_TPPBitsightGUID;
		let iID_TPPBitsightVRMLifecycle;
		let iID_TPPBitsightVRMLastUpdatedDate;
		let iID_TPPBitsightVRMRequirementsCompletion;
		let iID_TPPBitsightVRMRequirementsDueDate;
		let iID_TPPBitsightVRMReviewYear;
		let iID_TPPBitsightVRMTrustScore;
		let iID_TPPBitsightVRMRiskScore;

		let sName;
		let sID;
		//let sAlias;
		//Iterate through the FieldDefinition to get the field ids that we care about
		for (let h in resultJSON.Records.Metadata[0].FieldDefinitions[0].FieldDefinition) {
			sName = resultJSON.Records.Metadata[0].FieldDefinitions[0].FieldDefinition[h].$.name;
			//sAlias = resultJSON.Records.Metadata[0].FieldDefinitions[0].FieldDefinition[h].$.alias;
			sID = resultJSON.Records.Metadata[0].FieldDefinitions[0].FieldDefinition[h].$.id;

			//LogInfo("  Alias:" + sAlias + "   ID: " + sID);
			if (sName == "Bitsight VRM Domain") {
				iID_TPPBitsightVRMDomain = sID;
			} else if (sName == "Bitsight VRM Tags") {
				iID_TPPBitsightVRMTags = sID;
			} else if (sName == "Bitsight VRM Deep Link") {
				iID_TPPBitsightVRMDeepLink = sID;
			} else if (sName == "Bitsight Security Rating") {
				iID_TPPBitsightSecurityRating = sID;
			} else if (sName == "Bitsight VRM GUID") {
				iID_TPPBitsightVRMGUID = sID;
			} else if (sName == "Bitsight VRM Impact Score") {
				iID_TPPBitsightVRMImpactScore = sID;
			} else if (sName == "Bitsight GUID") {
				iID_TPPBitsightGUID = sID;
			} else if (sName == "Bitsight VRM Lifecycle") {
				iID_TPPBitsightVRMLifecycle = sID;
			} else if (sName == "Bitsight VRM Last Updated Date") {
				iID_TPPBitsightVRMLastUpdatedDate = sID;
			} else if (sName == "Bitsight VRM Requirements Completion %") {
				iID_TPPBitsightVRMRequirementsCompletion = sID;
			} else if (sName == "Bitsight VRM Requirements Due Date") {
				iID_TPPBitsightVRMRequirementsDueDate = sID;
			} else if (sName == "Bitsight VRM Review Year") {
				iID_TPPBitsightVRMReviewYear = sID;
			} else if (sName == "Bitsight VRM Trust Score") {
				iID_TPPBitsightVRMTrustScore = sID;
			} else if (sName == "Bitsight VRM Risk Score") {
				iID_TPPBitsightVRMRiskScore = sID;
			}
		}

		//TODO: Also need to get the values list values of all the values lists. Will need to manage the values lists by adding new ones too.
		//Tags will be the most challenging due to the number of changes and multiple values.
		//If the tag list value changes in Bitsight, it will create a duplicate in Archer, but Archer will always look correctly mapped. The only downside is that there will be extra unused values in the values list when doing reporting.
		//VRM Lifecycle is the only other list for now which is a guid in Bitsight as well, but single-select

		let sTPPContentID = "";
		let sTPPBitsightVRMDomain;
		let sTPPBitsightVRMTags;
		let sTPPBitsightVRMDeepLink;
		let sTPPBitsightSecurityRating;
		let sTPPBitsightVRMGUID;
		let sTPPBitsightVRMImpactScore;
		let sTPPBitsightGUID;
		let sTPPBitsightVRMLifecycle;
		let sTPPBitsightVRMLastUpdatedDate;
		let sTPPBitsightVRMRequirementsCompletion;
		let sTPPBitsightVRMRequirementsDueDate;
		let sTPPBitsightVRMReviewYear;
		let sTPPBitsightVRMTrustScore;
		let sTPPBitsightVRMRiskScore;

		//Iterate through each Archer TPP record....
		for (let i in resultJSON.Records.Record) {
			LogInfo("-----ARCHER TPP RECORD #" + i + "-------");
			sTPPContentID = resultJSON.Records.Record[i].$.contentId;
			sTPPBitsightVRMDomain = "";
			sTPPBitsightVRMTags = [];
			sTPPBitsightVRMDeepLink = "";
			sTPPBitsightSecurityRating = "";
			sTPPBitsightVRMGUID = "";
			sTPPBitsightVRMImpactScore = "";
			sTPPBitsightGUID = "";
			sTPPBitsightVRMLifecycle = "";
			sTPPBitsightVRMLastUpdatedDate = "";
			sTPPBitsightVRMRequirementsCompletion = "";
			sTPPBitsightVRMRequirementsDueDate = "";
			sTPPBitsightVRMReviewYear = "";
			sTPPBitsightVRMTrustScore = "";
			sTPPBitsightVRMRiskScore = "";

			//Iterate through the Field elements for the current config record to get the goodies
			for (let y in resultJSON.Records.Record[i].Field) {
				//Get the id of the field because we need to match on the ones we care about...
				sID = resultJSON.Records.Record[i].Field[y].$.id;

				//Now find all the good data we care about...
				if (sID == iID_TPPBitsightVRMDomain) {
					sTPPBitsightVRMDomain = GetArcherText(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_TPPBitsightVRMTags) {
					sTPPBitsightVRMTags = GetArcherValues(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_TPPBitsightVRMDeepLink) {
					sTPPBitsightVRMDeepLink = GetArcherText(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_TPPBitsightSecurityRating) {
					sTPPBitsightSecurityRating = GetArcherText(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_TPPBitsightVRMGUID) {
					sTPPBitsightVRMGUID = GetArcherText(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_TPPBitsightVRMImpactScore) {
					sTPPBitsightVRMImpactScore = GetArcherText(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_TPPBitsightGUID) {
					sTPPBitsightGUID = GetArcherText(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_TPPBitsightVRMLifecycle) {
					sTPPBitsightVRMLifecycle = GetArcherValue(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_TPPBitsightVRMLastUpdatedDate) {
					sTPPBitsightVRMLastUpdatedDate = GetArcherText(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_TPPBitsightVRMRequirementsCompletion) {
					sTPPBitsightVRMRequirementsCompletion = GetArcherText(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_TPPBitsightVRMRequirementsDueDate) {
					sTPPBitsightVRMRequirementsDueDate = GetArcherText(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_TPPBitsightVRMReviewYear) {
					sTPPBitsightVRMReviewYear = GetArcherText(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_TPPBitsightVRMTrustScore) {
					sTPPBitsightVRMTrustScore = GetArcherText(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_TPPBitsightVRMRiskScore) {
					sTPPBitsightVRMRiskScore = GetArcherText(resultJSON.Records.Record[i].Field[y]);
				}
			}

			LogInfo("Content ID: " + sTPPContentID + " sTPPBitsightVRMDomain: " + sTPPBitsightVRMDomain);
			//Populate the main record with the details we care about....
			aArcherTPPReport[aArcherTPPReport.length] = {
				"ArcherContentID": sTPPContentID,
				"BitsightVRMDomain": sTPPBitsightVRMDomain,
				"sTPPBitsightVRMTags": sTPPBitsightVRMTags,
				"sTPPBitsightVRMDeepLink": sTPPBitsightVRMDeepLink,
				"sTPPBitsightSecurityRating": sTPPBitsightSecurityRating,
				"sTPPBitsightVRMGUID": sTPPBitsightVRMGUID,
				"sTPPBitsightVRMImpactScore": sTPPBitsightVRMImpactScore,
				"sTPPBitsightGUID": sTPPBitsightGUID,
				"sTPPBitsightVRMLifecycle": sTPPBitsightVRMLifecycle,
				"sTPPBitsightVRMLastUpdatedDate": sTPPBitsightVRMLastUpdatedDate,
				"sTPPBitsightVRMRequirementsCompletion": sTPPBitsightVRMRequirementsCompletion,
				"sTPPBitsightVRMRequirementsDueDate": sTPPBitsightVRMRequirementsDueDate,
				"sTPPBitsightVRMReviewYear": sTPPBitsightVRMReviewYear,
				"sTPPBitsightVRMTrustScore": sTPPBitsightVRMTrustScore,
				"sTPPBitsightVRMRiskScore": sTPPBitsightVRMRiskScore,
				"APIStatus": "Ready",
			};

			LogInfo("*TPP Number#" + aArcherTPPReport.length + "=" + JSON.stringify(aArcherTPPReport[aArcherTPPReport.length - 1]));
		}

		//Save to file...
		if (bVerboseLogging === true) {
			var fs = require("fs");
			fs.writeFileSync("logs\\" + appPrefix + "\\" + "aArcherTPPReport.json", JSON.stringify(aArcherTPPReport));
			LogInfo("Saved to logs\\" + appPrefix + "\\" + "aArcherTPPReport.json file");
		}
	} catch (ex) {
		bSuccess = false;
		LogError("parseArcherTPPRecords() Error parsing Archer data. ex: " + ex);
		LogVerbose("parseArcherTPPRecords() Error parsing Archer data. ex.stack: " + ex.stack);
	}

	return bSuccess;
}

async function BitsightVRMMonitorVendor(i) {
	LogInfo("BitsightVRMMonitorVendor() Start. i=" + i + " Domain=" + aArcherTPPReport[i].BitsightVRMDomain);
	let bSuccess = true;

	try {
		const sURL = params["Bitsight_webroot"] + params["Bitsight_addMonitoredVendor"]; //Should be: https://service.Bitsighttech.com/customer-api/vrm/v1/vendors/monitored

		const sAuth = "Basic " + makeBasicAuth(params["Bitsight_token"]);

		const sDomain = aArcherTPPReport[i].BitsightVRMDomain;

		const headers = {
			"Content-Type": "application/json",
			"Accept": "application/json",
			"Authorization": sAuth,
			"X-Bitsight-CONNECTOR-NAME-VERSION": COST_ArcherBitsightVersion,
			"X-Bitsight-CALLING-PLATFORM-VERSION": COST_Platform,
			"X-Bitsight-CUSTOMER": BitsightCustomerName,
		};

		const options = {
			method: "POST",
			url: sURL,
			headers: headers,
			data: {
				domain: sDomain,
			},
		};

		LogInfo("BitsightVRMMonitorVendor() API options=" + JSON.stringify(options));

		try {
			const { data } = await axios.request(options);
			LogInfo("BitsightVRMMonitorVendor() data=" + data);
			if (data == null) {
				LogInfo("BitsightVRMMonitorVendor() API returned null which is expected for a successful call to this endpoint.");
				aArcherTPPReport[i].APIStatus = "Successfully added to Bitsight.";
				totalReportSuccess++;
			} else {
				LogWarn("BitsightVRMMonitorVendor() API did NOT return null which is expected for a successful call to this endpoint.");
			}
		} catch (ex) {
			totalReportErrors++;
			//Note that calls to add a vendor that exists will return a 400 error message
			//Not sure if other situations return 400 as well though. 401 for auth error tested successfully.
			if (ex == "AxiosError: Request failed with status code 400") {
				LogWarn(
					"BitsightVRMMonitorVendor(): HTTP 400 error code adding domain " +
						aArcherTPPReport[i].BitsightVRMDomain +
						" - This is expected if the vendor exists in Bitsight VRM. Not sure if 400 errors are for other situations."
				);
				aArcherTPPReport[i].APIStatus = "Warning: Vendor may already exist or other error.";
			} else {
				LogError("BitsightVRMMonitorVendor() Axios call error. ex: " + ex);
				LogVerbose("BitsightVRMMonitorVendor() Axios call error. ex.stack: " + ex.stack);
				aArcherTPPReport[i].APIStatus = "Error adding to Bitsight VRM.";
				return false;
			}
		}
	} catch (ex) {
		aArcherTPPReport[i].APIStatus = "Error constructing api call to Bitsight VRM.";
		LogError("BitsightVRMMonitorVendor() Error constructing api call. ex: " + ex);
		LogVerbose("BitsightVRMMonitorVendor() Error constructing api call. ex.stack: " + ex.stack);
		return false;
	}
	return bSuccess;
}

async function getBitsightListValues() {
	LogInfo("getBitsightListValues() Start.");
	let bSuccess = true;

	bSuccess = await getBitsightListValuesTags();
	if (bSuccess === true) {
		bSuccess = await getBitsightListValuesLifecycles();
	}

	return bSuccess;
}

async function getBitsightListValuesTags() {
	LogInfo("getBitsightListValuesTags() Start.");
	let bSuccess = true;

	try {
		const sURL = params["Bitsight_webroot"] + params["Bitsight_getcompanytags"]; //Should be: https://service.Bitsighttech.com/customer-api/vrm/v1/tags

		const sAuth = "Basic " + makeBasicAuth(params["Bitsight_token"]);

		const headers = {
			"Content-Type": "application/json",
			"Accept": "application/json",
			"Authorization": sAuth,
			"X-Bitsight-CONNECTOR-NAME-VERSION": COST_ArcherBitsightVersion,
			"X-Bitsight-CALLING-PLATFORM-VERSION": COST_Platform,
			"X-Bitsight-CUSTOMER": BitsightCustomerName,
		};

		const options = {
			method: "GET",
			url: sURL,
			headers: headers,
			data: {},
		};

		LogVerbose("getBitsightListValuesTags() API options=" + JSON.stringify(options));

		try {
			const { data } = await axios.request(options);
			//LogVerbose("getBitsightListValuesTags() data=" + JSON.stringify(data));

			if (typeof data == "undefined" || data == null) {
				LogError("getBitsightListValuesTags() API returned undefined or null.");
				bSuccess = false;
			} else {
				//Iterate the tags and populate the object

				for (let i in data) {
					//LogVerbose("getBitsightListValuesTags() i=" + i + " " + data[i].name + "=" + data[i].guid);

					BitsightVRMTags[BitsightVRMTags.length] = {
						"name": data[i].name,
						"guid": data[i].guid,
					};
				}

				//Note that it is possible that there are no tags. Our Bitsight instance started without any.

				//Save to file...
				if (bVerboseLogging === true) {
					var fs = require("fs");
					fs.writeFileSync("logs\\" + appPrefix + "\\" + "BitsightVRMTags.json", JSON.stringify(BitsightVRMTags));
					LogInfo("getBitsightListValuesTags() Saved to logs\\" + appPrefix + "\\" + "BitsightVRMTags.json file");
				}
			}
		} catch (ex) {
			LogError("getBitsightListValuesTags() Axios call error. ex: " + ex);
			LogVerbose("getBitsightListValuesTags() Axios call error. ex.stack: " + ex.stack);
			bSuccess = false;
		}
	} catch (ex) {
		LogError("getBitsightListValuesTags() Error constructing api call. ex: " + ex);
		LogVerbose("getBitsightListValuesTags() Error constructing api call. ex.stack: " + ex.stack);
		bSuccess = false;
	}
	return bSuccess;
}

async function getBitsightListValuesLifecycles() {
	LogInfo("getBitsightListValuesLifecycles() Start.");
	let bSuccess = true;

	try {
		const sURL = params["Bitsight_webroot"] + params["Bitsight_getcompanylifecycles"]; //Should be: https://service.Bitsighttech.com/customer-api/vrm/v1/life-cycle-stages

		const sAuth = "Basic " + makeBasicAuth(params["Bitsight_token"]);

		const headers = {
			"Content-Type": "application/json",
			"Accept": "application/json",
			"Authorization": sAuth,
			"X-Bitsight-CONNECTOR-NAME-VERSION": COST_ArcherBitsightVersion,
			"X-Bitsight-CALLING-PLATFORM-VERSION": COST_Platform,
			"X-Bitsight-CUSTOMER": BitsightCustomerName,
		};

		const options = {
			method: "GET",
			url: sURL,
			headers: headers,
			data: {},
		};

		LogVerbose("getBitsightListValuesLifecycles() API options=" + JSON.stringify(options));

		try {
			const { data } = await axios.request(options);
			//LogVerbose("getBitsightListValuesLifecycles() data=" + JSON.stringify(data));

			if (typeof data == "undefined" || data == null) {
				LogError("getBitsightListValuesLifecycles() API returned undefined or null.");
				bSuccess = false;
			} else {
				//Iterate the tags and populate the object

				for (let i in data) {
					//LogVerbose("getBitsightListValuesLifecycles() i=" + i + " " + data[i].name + "=" + data[i].guid);

					BitsightVRMLifecycles[BitsightVRMLifecycles.length] = {
						"name": data[i].name, //Need to use "Name" because the "Status" only has two values (Pending and Completed). The Name is more customizable.
						"guid": data[i].guid,
						"status": data[i].status,
						//"order": data[i].order //not used yet, but holding onto it just in case
						//"color": data[i].color_name //not used yet, but holding onto it just in case
					};
				}

				//Save to file...
				if (bVerboseLogging === true) {
					var fs = require("fs");
					fs.writeFileSync("logs\\" + appPrefix + "\\" + "BitsightVRMLifecycles.json", JSON.stringify(BitsightVRMLifecycles));
					LogInfo("getBitsightListValuesLifecycles() Saved to logs\\" + appPrefix + "\\" + "BitsightVRMLifecycles.json file");
				}
			}
		} catch (ex) {
			LogError("getBitsightListValuesLifecycles() Axios call error. ex: " + ex);
			LogVerbose("getBitsightListValuesLifecycles() Axios call error. ex.stack: " + ex.stack);
			bSuccess = false;
		}
	} catch (ex) {
		LogError("getBitsightListValuesLifecycles() Error constructing api call. ex: " + ex);
		LogVerbose("getBitsightListValuesLifecycles() Error constructing api call. ex.stack: " + ex.stack);
		bSuccess = false;
	}
	return bSuccess;
}

async function ValidateAndUpdateListValues() {
	LogInfo("ValidateAndUpdateListValues() Start - Tags.");
	let bSuccess = true;
	try {
		//Check Bitsight Tags - iterate and if new values are found, add them to Archer
		for (let b in BitsightVRMTags) {
			let sBitsightTagValue = BitsightVRMTags[b].name;
			LogVerbose("ValidateAndUpdateListValues() Looking for sBitsightTagValue=" + sBitsightTagValue);
			let sArcherValuesListID = null;

			//Archer Value List...Find the correct list
			for (let a in ArcherValuesLists) {
				if (ArcherValuesLists[a].FieldName == "Bitsight VRM Tags") {
					sArcherValuesListID = ArcherValuesLists[a].ValuesListID;
					LogVerbose("ValidateAndUpdateListValues()   Found 'Bitsight VRM Tags' in ArcherValuesLists");
					let bFound = false;
					//Archer Values in the list...
					for (let aValue in ArcherValuesLists[a].Values) {
						//Does the Bitsight value match the Archer list?
						let sArcherTagValue = ArcherValuesLists[a].Values[aValue].name;
						//LogVerbose("ValidateAndUpdateListValues()      Current value: sArcherTagValue=" + sArcherTagValue);
						if (sArcherTagValue == sBitsightTagValue) {
							//we need the Archer Values List ID to add the new value
							LogVerbose("ValidateAndUpdateListValues()      **MATCH FOUND** " + sBitsightTagValue);
							bFound = true;
							break;
						}
					}
					try {
						if (bFound === false) {
							LogVerbose("ValidateAndUpdateListValues()      **NOT FOUND** Add:" + sBitsightTagValue);
							let newValuesListID = await CreateArcherListValue(sArcherValuesListID, sBitsightTagValue);
							if (newValuesListID != null && newValuesListID > 0) {
								//Need to add to the Archer values list values
								ArcherValuesLists[a].Values[ArcherValuesLists[a].Values.length] = {
									"name": sBitsightTagValue,
									"id": newValuesListID,
								};
							}
						}
					} catch (ex) {
						LogWarn("ValidateAndUpdateListValues() Unable to update ArcherValuesLists with new values list value. ex: " + ex);
						LogVerbose("ValidateAndUpdateListValues() Unable to update ArcherValuesLists with new values list value. ex.stack: " + ex.stack);
					}
				}
			}
		}

		LogInfo("ValidateAndUpdateListValues() Start - Lifecycle.");
		//Check Lifecycle Values - iterate and if new values are found, add them to Archer

		//TODO: Update the variable names here! I copied from the code block above, but it's confusing when it references tags when it's the lifecycles.
		for (let b in BitsightVRMLifecycles) {
			let sBitsightTagValue = BitsightVRMLifecycles[b].name;
			LogVerbose("ValidateAndUpdateListValues() Looking for sBitsightTagValue=" + sBitsightTagValue);
			let sArcherValuesListID = null;

			//Archer Value List...Find the correct list
			for (let a in ArcherValuesLists) {
				if (ArcherValuesLists[a].FieldName == "Bitsight VRM Lifecycle") {
					sArcherValuesListID = ArcherValuesLists[a].ValuesListID;
					LogVerbose("ValidateAndUpdateListValues()   Found 'Bitsight VRM Lifecycle' in ArcherValuesLists");
					let bFound = false;
					//Archer Values in the list...
					for (let aValue in ArcherValuesLists[a].Values) {
						//Does the Bitsight value match the Archer list?
						let sArcherTagValue = ArcherValuesLists[a].Values[aValue].name;
						//LogVerbose("ValidateAndUpdateListValues()      Current value: sArcherTagValue=" + sArcherTagValue);
						if (sArcherTagValue == sBitsightTagValue) {
							//we need the Archer Values List ID to add the new value
							LogVerbose("ValidateAndUpdateListValues()      **MATCH FOUND** " + sBitsightTagValue);
							bFound = true;
							break;
						}
					}
					try {
						if (bFound === false) {
							LogVerbose("ValidateAndUpdateListValues()      **NOT FOUND** Add:" + sBitsightTagValue);
							let newValuesListID = await CreateArcherListValue(sArcherValuesListID, sBitsightTagValue);
							if (newValuesListID != null && newValuesListID > 0) {
								//Need to add to the Archer values list values
								ArcherValuesLists[a].Values[ArcherValuesLists[a].Values.length] = {
									"name": sBitsightTagValue,
									"id": newValuesListID,
								};
							}
						}
					} catch (ex) {
						LogWarn("ValidateAndUpdateListValues() Unable to update ArcherValuesLists with new values list value. ex: " + ex);
						LogVerbose("ValidateAndUpdateListValues() Unable to update ArcherValuesLists with new values list value. ex.stack: " + ex.stack);
					}
				}
			}
		}
	} catch (ex) {
		LogError("ValidateAndUpdateListValues() Error iterating loops. ex: " + ex);
		LogVerbose("ValidateAndUpdateListValues() Error iterating loops. ex.stack: " + ex.stack);
		bSuccess = false;
	}

	//Output the updated values list file for troubleshooting.
	if (bVerboseLogging === true) {
		var fs = require("fs");
		fs.writeFileSync("logs\\" + appPrefix + "\\" + "ArcherValuesListsUpdated.json", JSON.stringify(ArcherValuesLists));
		LogVerbose("Saved ArcherValuesLists to logs\\" + appPrefix + "\\" + "ArcherValuesListsUpdated.json file");
	}

	return bSuccess;
}

async function CreateArcherListValue(sListNumber, sNewValue) {
	LogInfo("CreateArcherListValue() Start. sListNumber=" + sListNumber + " sNewValue=" + sNewValue);
	let bSuccess = true;
	try {
		let sXML =
			'<?xml version="1.0" encoding="utf-8"?><soap:Envelope xmlns:xsi="http://www.w3.org/2001/XMLSchemainstance" xmlns:xsd="http://www.w3.org/2001/XMLSchema" xmlns:soap="http://schemas.xmlsoap.org/soap/envelope/">' +
			'<soap:Body><CreateValuesListValue xmlns="http://archer-tech.com/webservices/"><sessionToken>' +
			sArcherSessionToken +
			"</sessionToken><valuesListId>" +
			sListNumber +
			"</valuesListId><valuesListValueName>" +
			sNewValue +
			"</valuesListValueName></CreateValuesListValue></soap:Body></soap:Envelope>";

		//Must escape the XML to next inside of the soap request...
		//sXML = sXML.replace(/&/g, "&amp;").replace(/</g, "&lt;").replace(/>/g, "&gt;").replace(/"/g, "&quot;").replace(/'/g, "&apos;");

		LogVerbose("CreateArcherListValue() sXML=" + sXML);

		/* build url */
		var sUrl = params["archer_webroot"] + params["archer_ws_root"] + params["archer_fieldpath"];

		var headers = {
			"content-type": "text/xml; charset=utf-8",
			"SOAPAction": "http://archer-tech.com/webservices/CreateValuesListValue",
		};

		// const postBody =
		// 	'<?xml version="1.0" encoding="utf-8"?><soap:Envelope xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xmlns:xsd="http://www.w3.org/2001/XMLSchema" xmlns:soap="http://schemas.xmlsoap.org/soap/envelope/">' +
		// 	"<soap:Body>" +
		// 	sXML +
		// 	"</CreateValuesListValue></soap:Body></soap:Envelope>";

		const httpConfig = {
			method: "POST",
			url: sUrl,
			headers: headers,
			data: sXML,
		};

		//Execute API Query
		LogVerbose("CreateArcherListValue() API call httpConfig=" + JSON.stringify(httpConfig));
		let data;
		await axios(httpConfig)
			.then(function (res) {
				data = res.data;
				LogVerbose("CreateArcherListValue() Axios call complete. data=" + data);
			})
			.catch(function (error) {
				bSuccess = false;
				LogError("CreateArcherListValue() Axios call error. Err: " + error);
				LogVerbose("CreateArcherListValue() Axios call error. Err.stack: " + error.stack);
			});

		if (bSuccess === true) {
			try {
				//Convert XML data results to an XMLDOM for parsing
				let doc = xmlStringToXmlDoc(data);

				if (bSuccess === true && typeof doc != "undefined" && doc != null) {
					const resultElement = doc.getElementsByTagName("CreateValuesListValueResult")[0];

					if (resultElement) {
						const sArcherValuesListValueID = resultElement.textContent;
						LogInfo("CreateArcherListValue() Result Values List Value ID:" + sArcherValuesListValueID); // Output: Result Value: 81301
						//Success! Return the ID
						const iArcherValuesListValueID = parseInt(sArcherValuesListValueID, 10); // Radix 10 is important!
						return iArcherValuesListValueID;
					} else {
						LogError("CreateArcherListValue() element not found.");
						return null;
					}
				}
			} catch (ex) {
				LogError("CreateArcherListValue() Error parsing new Archer values list value. ex: " + ex);
				LogVerbose("CreateArcherListValue() Error parsing new Archer values list value. ex.stack: " + ex.stack);
				return null;
			}
		} else {
			return null;
		}
	} catch (ex) {
		LogError("CreateArcherListValue() Error constructing api call. ex: " + ex);
		LogVerbose("CreateArcherListValue() Error constructing api call. ex.stack: " + ex.stack);
		return null;
	}
	//If we got here, we didn't receive a new values list ID.
	return null;
}

async function BitsightListAllVendors() {
	LogInfo("BitsightListAllVendors() Start.");
	let bSuccess = true;

	try {
		const sURL = params["Bitsight_webroot"] + params["Bitsight_listallvendors"]; //Should be: https://service.Bitsighttech.com/customer-api/vrm/v1/vendors/query

		const sAuth = "Basic " + makeBasicAuth(params["Bitsight_token"]);

		const headers = {
			"Content-Type": "application/json",
			"Accept": "application/json",
			"Authorization": sAuth,
			"X-Bitsight-CONNECTOR-NAME-VERSION": COST_ArcherBitsightVersion,
			"X-Bitsight-CALLING-PLATFORM-VERSION": COST_Platform,
			"X-Bitsight-CUSTOMER": BitsightCustomerName,
		};

		const postBody = {
			"q": "",
		};

		const options = {
			method: "POST",
			url: sURL,
			headers: headers,
			data: postBody,
		};

		LogVerbose("BitsightListAllVendors() API options=" + JSON.stringify(options));

		try {
			const { data } = await axios.request(options);

			//LogVerbose("BitsightListAllVendors() data=" + data);
			//Output the data to a file for troubleshooting.
			if (bVerboseLogging === true) {
				var fs = require("fs");
				fs.writeFileSync("logs\\" + appPrefix + "\\" + "BitsightVRMListAllVendors.json", JSON.stringify(data));
				LogVerbose("Saved BitsightVRMListAllVendors to logs\\" + appPrefix + "\\" + "BitsightVRMListAllVendors.json file");
			}

			if (typeof data != "undefined" && data != null && typeof data.results != "undefined" && data.results != null && data.results.length > 0) {
				LogInfo("BitsightListAllVendors() API returned data.");

				return await ParseBitsightVRMVendors(data);
			} else {
				LogWarn("BitsightListAllVendors() API returned null data. No companies in Bitsight VRM? Check the BitsightVRMListAllVendors.json file for output.");
				return false;
			}
		} catch (ex) {
			LogError("BitsightListAllVendors() Axios call error. ex: " + ex);
			LogVerbose("BitsightListAllVendors() Axios call error. ex.stack: " + ex.stack);
			return false;
		}
	} catch (ex) {
		LogError("BitsightListAllVendors() Error constructing api call. ex: " + ex);
		LogVerbose("BitsightListAllVendors() Error constructing api call. ex.stack: " + ex.stack);
		return false;
	}

	//Shouldn't get here, but if we do, return bSuccess
	return bSuccess;
}

async function ParseBitsightVRMVendors(data) {
	//Goal: Populate the BitsightVRMVendors[] object with vendors from VRM.
	//Data format documentation: https://Bitsight.stoplight.io/docs/VRM-api/r93i8xhwr9odb-list-all-vendors
	LogInfo("ParseBitsightVRMVendors() Start");
	let bSuccess = true;
	try {
		LogInfo("ParseBitsightVRMVendors() Bitsight Vendor records found:" + data.results.length);

		//For each result element, parse the data and add to our main object.
		//Need to make additional API calls per vendor to get the remaining data.
		for (let i in data.results) {
			try {
				//Obtain and validate data elements
				let sBSVRMDomain = getString(data.results[i].company_domain);
				LogInfo("ParseBitsightVRMVendors() ----------------------------------NEXT VENDOR--------------------------------------------");
				LogInfo("ParseBitsightVRMVendors() Parsing domain: " + sBSVRMDomain);

				let iArcherContentID = getArcherContentID(sBSVRMDomain);
				if (iArcherContentID != null) {
					let sBSVRMVendorGUID = getString(data.results[i].vendor_guid); //Vendor GUID is used to lookup an individual vendor. We need this to get the Bitsight Ratings
					let bBSIsManaged = getBoolean(data.results[i].is_managed);
					let sBSGUID = await getBitsightVendorGuid(getString(sBSVRMVendorGUID), bBSIsManaged); //Uses the Vendor GUID to get the bst_entity_guid so we can then lookup the Bitsight Rating
					let sBSVRMTags = lookupBitsightTags(data.results[i].tags_guids); //Get Tags and align to Archer IDs.
					let sBSSecurityRating = await getBitsightRating(sBSGUID); //Uses the sBSGUID (bst_entity_guid) to get the Bitsight Rating via an extra API call.
					let sBSVRMGUID = getString(data.results[i].connection_guid);
					let sBSVRMImpactScore = getInt(data.results[i].impact_score);
					let sBSVRMLifecycle = lookupBitsightLifecycle(getString(data.results[i].life_cycle_stage_guid)); //Lookup the guid and get the values list id for Archer
					let sBSVRMLastUpdatedDate = getDateYYYYMMMDD(); //Assuming current system date for now. Not sure if there is another field we need to use.
					let sBSVRMRequirementsCompletion = 0;
					if (bBSIsManaged === true) {
						sBSVRMRequirementsCompletion = await getBitsightRequirementProgress(sBSVRMVendorGUID); //Another API call to get the progres using the Vendor GUID.
					}
					let sBSVRMRequirementsDueDate = getDate(data.results[i].due_date);
					let sBSVRMReviewYear = getString(data.results[i].review_year);
					let sBSVRMTrustScore = getInt(data.results[i].trust_score);
					let sBSVRMRiskScore = getInt(data.results[i].risk_score);

					let sBSVRMDeepLink = makeBitsightVendorURL(sBSVRMVendorGUID, bBSIsManaged); //Generate URL based on the guid and whether the vendor is managed or monitored

					//put into the BitsightVRMVendors[] clean object

					BitsightVRMVendors[BitsightVRMVendors.length] = {
						"VRMDomain": sBSVRMDomain,
						"iArcherContentID": iArcherContentID,
						"VRMTags": sBSVRMTags,
						"VRMDeepLink": sBSVRMDeepLink,
						"SecurityRating": sBSSecurityRating,
						"VRMGUID": sBSVRMGUID,
						"VendorGUID": sBSVRMVendorGUID,
						"VRMImpactScore": sBSVRMImpactScore,
						"BSGUID": sBSGUID,
						"VRMLifecycle": sBSVRMLifecycle,
						"VRMLastUpdatedDate": sBSVRMLastUpdatedDate,
						"VRMRequirementsCompletion": sBSVRMRequirementsCompletion,
						"VRMRequirementsDueDate": sBSVRMRequirementsDueDate,
						"VRMReviewYear": sBSVRMReviewYear,
						"VRMTrustScore": sBSVRMTrustScore,
						"VRMRiskScore": sBSVRMRiskScore,
						"VRMIsManaged": bBSIsManaged,
						"APIStatus": "01-Bitsight Data Ready.",
					};
				} else {
					LogWarn("ParseBitsightVRMVendors() Vendor Exists in Bitsight, but not in Archer. Bitsight Domain=" + sBSVRMDomain);
				}
			} catch (ex) {
				LogError("ValidateAndUpdateListValues() Error parsing. domain=" + data.results[i].company_domain + " ex: " + ex);
				LogVerbose("ValidateAndUpdateListValues() Error parsing. ex.stack: " + ex.stack);
				bSuccess = false;
			}
		}
	} catch (ex) {
		LogError("ValidateAndUpdateListValues() Error iterating loop.  ex: " + ex);
		LogVerbose("ValidateAndUpdateListValues() Error iterating loop. ex.stack: " + ex.stack);
		bSuccess = false;
	}

	//Output BitsightVRMVendors for troubleshooting.
	if (bSuccess === true && bVerboseLogging === true) {
		var fs = require("fs");
		fs.writeFileSync("logs\\" + appPrefix + "\\" + "BitsightVRMVendors.json", JSON.stringify(BitsightVRMVendors));
		LogVerbose("Saved BitsightVRMVendors to logs\\" + appPrefix + "\\" + "BitsightVRMVendors.json file");
	}

	return bSuccess;
}

function getString(txt) {
	try {
		if (typeof txt == "undefined" || txt == null) {
			return "";
		} else {
			//attempt to trim to remove spaces.
			try {
				let sTrimmed = txt.toString().trim();
				return sTrimmed;
			} catch (ex) {
				LogWarn("getString() Unable to trim the text. Returning empty string. ex:" + ex);
				LogVerbose("getString() Unable to trim the text. Returning empty string. ex.stack:" + ex.stack);
				return "";
			}
		}
	} catch (ex) {
		LogWarn("getString() Issue parsing the string/text provided. Returning empty string. ex:" + ex);
		LogVerbose("getString() Issue parsing the string/text provided. Returning empty string. ex.stack:" + ex.stack);
		return "";
	}
}

function getInt(txt) {
	//While we are expecting a float from the JSON, let's be extra careful by validating and parse it as a float just in case.
	try {
		if (typeof txt == "undefined" || txt == null) {
			return "0";
		} else {
			//attempt to convert to int, roundup, and convert back to string
			try {
				const num = parseFloat(txt);

				if (isNaN(num)) {
					LogWarn("getInt() parseFloat could not return a number. Returning empty string. ex:" + ex);
					return "";
				}

				const rounded = Math.round(num);
				return rounded; //Note that Archer is expecting a numeric value when updating a record.
			} catch (ex) {
				LogWarn("getInt() Unable to trim the text. Returning empty string. ex:" + ex);
				LogVerbose("getInt() Unable to trim the text. Returning empty string. ex.stack:" + ex.stack);
				return "";
			}
		}
	} catch (ex) {
		LogWarn("getInt() Issue parsing the string/text provided. Returning empty string. ex:" + ex);
		LogVerbose("getInt() Issue parsing the string/text provided. Returning empty string. ex.stack:" + ex.stack);
		return "";
	}
}

function getBoolean(txt) {
	try {
		if (typeof txt == "undefined" || txt == null) {
			//Assume false
			return false;
		} else {
			try {
				if (txt === true || txt == "true") {
					return true;
				} else {
					return false;
				}
			} catch (ex) {
				LogWarn("getBoolean() Unable to determine true/false. Returning false. ex:" + ex);
				LogVerbose("getBoolean() Unable to determine true/false. Returning false. ex.stack:" + ex.stack);
				return false;
			}
		}
	} catch (ex) {
		LogWarn("getBoolean() Issue parsing the string/text provided. Returning false. ex:" + ex);
		LogVerbose("getBoolean() Issue parsing the string/text provided. Returning falseg. ex.stack:" + ex.stack);
		return false;
	}
}

function getDate(txt) {
	//Input type will be a string
	//Date format should be: YYYY-MM-DD
	//Example: "2025-04-07"
	try {
		if (typeof txt == "undefined" || txt == null) {
			return "";
		} else {
			//attempt to trim to remove spaces.
			try {
				let sTrimmed = txt.toString().trim();
				return sTrimmed;
			} catch (ex) {
				LogWarn("getDate() Unable to trim the text. Returning empty string. ex:" + ex);
				LogVerbose("getDate() Unable to trim the text. Returning empty string. ex.stack:" + ex.stack);
				return "";
			}
		}
	} catch (ex) {
		LogWarn("getDate() Issue parsing the string/text provided. Returning empty string. ex:" + ex);
		LogVerbose("getDate() Issue parsing the string/text provided. Returning empty string. ex.stack:" + ex.stack);
		return "";
	}
}

function makeBitsightVendorURL(sBSVRMGUID, bBSIsManaged) {
	LogVerbose("makeBitsightVendorURL() Start. sBSVRMGUID=" + sBSVRMGUID + " bBSIsManaged=" + bBSIsManaged);
	try {
		//validate the parameters first
		if (typeof sBSVRMGUID == "undefined" || sBSVRMGUID == null || sBSVRMGUID == "" || typeof bBSIsManaged == "undefined" || bBSIsManaged == null) {
			LogWarn("makeBitsightVendorURL() Cannot generate URL to portal. VRM GUID or bIsManaged empty. Returning empty string.");
			return "";
		} else {
			//URL is based on whether it's Managed or Monitored
			//Monitored example: https://service.Bitsighttech.com/app/vrm/profile/monitored/4c81e8c0-abe4-49ba-9079-89202e51759e/overview/
			//Managed example: https://service.Bitsighttech.com/app/vrm/profile/managed/96f2d60f-e86a-45ad-b5b0-ce33e1538377/overview

			let url = "";
			if (bBSIsManaged === true || bBSIsManaged == "true") {
				url = params["Bitsight_webroot"] + params["Bitsight_managedvendorpath"] + sBSVRMGUID + params["Bitsight_overviewpath"];
				url;
			} else {
				url = params["Bitsight_webroot"] + params["Bitsight_managedvendorpath"] + sBSVRMGUID + params["Bitsight_overviewpath"];
				url;
			}

			url = "<a href='" + url + "' target='_blank'>" + url + "</a>";
			return url;
		}
	} catch (ex) {
		LogWarn("makeBitsightVendorURL() Issue generating the Bitsight portal url for this vendor. Returning empty string. ex:" + ex);
		LogVerbose("makeBitsightVendorURL() Issue generating the Bitsight portal url for this vendor. Returning empty string. ex.stack:" + ex.stack);
		return "";
	}
}

function lookupBitsightLifecycle(life_cycle_stage_guid) {
	//Note: Do not use the "status" value from Bitsight..There are currently only 2 values (Pending and Completed). The "name" is the more descript value to show.
	LogVerbose("lookupBitsightLifecycle() start. guid=" + life_cycle_stage_guid);

	try {
		if (typeof life_cycle_stage_guid == "undefined" || life_cycle_stage_guid == null || life_cycle_stage_guid == "") {
			LogWarn("lookupBitsightLifecycle() life_cycle_stage_guid was empty. Returning emptyset.");
			return {};
		} else {
			//iterate and find the guid and get the info
			let bsLCName = "";
			let bFound = false;
			for (let bsLC in BitsightVRMLifecycles) {
				if (life_cycle_stage_guid == BitsightVRMLifecycles[bsLC].guid) {
					bsLCName = BitsightVRMLifecycles[bsLC].name;
					bFound = true;
					break;
				}
			}
			LogVerbose("lookupBitsightLifecycle() bsLCName=" + bsLCName);

			if (bFound === true) {
				let archerValuesListID = "";
				bFound = false;
				//Now go get the Archer valueslist ID
				for (let aVL in ArcherValuesLists) {
					if (ArcherValuesLists[aVL].FieldName == "Bitsight VRM Lifecycle") {
						for (let aVal in ArcherValuesLists[aVL].Values) {
							if (ArcherValuesLists[aVL].Values[aVal].name == bsLCName) {
								archerValuesListID = ArcherValuesLists[aVL].Values[aVal].id;
								bFound = true;
								break;
							}
						}
					}
				}
				LogVerbose("lookupBitsightLifecycle() archerValuesListID=" + archerValuesListID);
				if (bFound === true) {
					//Populate the value
					let lifecycleinfo = {
						"Bitsight_life_cycle_stage_guid": life_cycle_stage_guid,
						"Name": bsLCName,
						"ArcherLifecyclestagelistID": archerValuesListID,
					};
					return lifecycleinfo;
				}
			}
		}
	} catch (ex) {
		LogWarn("lookupBitsightLifecycle() life_cycle_stage_guid error parsing. Returning emptyset. ex:" + ex);
		LogVerbose("lookupBitsightLifecycle() life_cycle_stage_guid error parsing. Returning emptyset. ex.stack:" + ex.stack);
		return {};
	}

	//If we got here, return blank because we don't have the info we need.
	LogWarn("lookupBitsightLifecycle() life_cycle_stage_guid did not match. returning emptyset.");
	return {};
}

function lookupBitsightTags(tags_guids) {
	//Lookup the tags (if any) for this vendor and return an object with the guid, name, and archer values list id
	//tags_guids will be an array of strings
	//Example: 	"tags_guids": [
	//				"174baace-4fad-465d-a43e-b6a21d9bce6c",
	//				"2834e259-a80c-45ad-b5bb-adbc56e1f006"
	//			],

	LogVerbose("lookupBitsightTags() start. tags_guids=" + JSON.stringify(tags_guids));

	try {
		if (typeof tags_guids == "undefined" || tags_guids == null || tags_guids.length == 0) {
			LogVerbose("lookupBitsightTags() tags_guids was empty. Returning emptyset.");
			return [];
		} else {
			let VRMTags = [];
			//Iterate the array of strings
			for (let t in tags_guids) {
				let tag_guid = tags_guids[t];

				//iterate and find the guid and get the info
				let bsTagName = "";
				let bFound = false;
				for (let bsTag in BitsightVRMTags) {
					if (tag_guid == BitsightVRMTags[bsTag].guid) {
						bsTagName = BitsightVRMTags[bsTag].name;
						bFound = true;
						break;
					}
				}
				LogVerbose("lookupBitsightTags() bsTagName=" + bsTagName);

				if (bFound === true) {
					let archerValuesListID = "";
					bFound = false;
					//Now go get the Archer valueslist ID
					for (let aVL in ArcherValuesLists) {
						if (ArcherValuesLists[aVL].FieldName == "Bitsight VRM Tags") {
							for (let aVal in ArcherValuesLists[aVL].Values) {
								if (ArcherValuesLists[aVL].Values[aVal].name == bsTagName) {
									archerValuesListID = ArcherValuesLists[aVL].Values[aVal].id;
									bFound = true;
									break;
								}
							}
						}
					}
					LogVerbose("lookupBitsightTags() archerValuesListID=" + archerValuesListID);
					if (bFound === true) {
						//Populate the value
						let tagInfo = {
							"Bitsight_tag_guid": tag_guid,
							"Name": bsTagName,
							"ArchertaglistID": archerValuesListID,
						};

						VRMTags[VRMTags.length] = tagInfo;
					}
				}
			}

			return VRMTags;
		}
	} catch (ex) {
		LogWarn("lookupBitsightTags() tags_guids error parsing. Returning emptyset. ex:" + ex);
		LogVerbose("lookupBitsightTags() tags_guids error parsing. Returning emptyset. ex.stack:" + ex.stack);
		return [];
	}

	//If we got here, return blank because we don't have the info we need. Technically this is unreachable code.
	LogWarn("lookupBitsightTags() tags_guids empty or no matches. returning emptyset.");
	return [];
}

async function getBitsightVendorGuid(sBSVRMVendorGUID, bBSIsManaged) {
	LogVerbose("getBitsightVendorGuid() start.");
	//Goal is to make API call to Bitsight VRM to get the bst_entity_guid.
	//API call is based on whether the vendor is managed or monitored
	try {
		if (typeof sBSVRMVendorGUID == "undefined" || sBSVRMVendorGUID == "" || sBSVRMVendorGUID == null || typeof bBSIsManaged == "undefined") {
			LogInfo("getBitsightVendorGuid() sBSVRMVendorGUID or bBSIsManaged empty. Returning empty string.");
			return "";
		}

		let sURL = "";
		if (bBSIsManaged === true) {
			sURL = params["Bitsight_webroot"] + params["Bitsight_managedvendorpathapi"] + sBSVRMVendorGUID; //Example:https://service.Bitsighttech.com/customer-api/vrm/v1/vendors/managed/385db583-5c09-4a11-9df8-eee0346d3a72
		} else {
			sURL = params["Bitsight_webroot"] + params["Bitsight_monitoredvendorpathapi"] + sBSVRMVendorGUID; //Example:https://service.Bitsighttech.com/customer-api/vrm/v1/vendors/monitored/385db583-5c09-4a11-9df8-eee0346d3a72
		}
		const sAuth = "Basic " + makeBasicAuth(params["Bitsight_token"]);

		const headers = {
			"Content-Type": "application/json",
			"Accept": "application/json",
			"Authorization": sAuth,
			"X-Bitsight-CONNECTOR-NAME-VERSION": COST_ArcherBitsightVersion,
			"X-Bitsight-CALLING-PLATFORM-VERSION": COST_Platform,
			"X-Bitsight-CUSTOMER": BitsightCustomerName,
		};

		const options = {
			method: "GET",
			url: sURL,
			headers: headers,
			data: {},
		};

		LogVerbose("getBitsightVendorGuid() API options=" + JSON.stringify(options));

		try {
			const { data } = await axios.request(options);
			//LogVerbose("getBitsightVendorGuid() data=" + JSON.stringify(data));

			if (typeof data == "undefined" || data == null) {
				LogError("getBitsightVendorGuid() API returned undefined or null.");
				return "";
			} else {
				//Iterate the tags and populate the object

				if (typeof data.bst_entity_guid == "undefined" || data.bst_entity_guid == null || data.bst_entity_guid == "") {
					LogInfo("getBitsightVendorGuid() API returned undefined or null for bst_entity_guid.");
					return "";
				} else {
					LogVerbose("getBitsightVendorGuid() bst_entity_guid=" + data.bst_entity_guid);
					return data.bst_entity_guid;
				}
			}
		} catch (ex) {
			LogError("getBitsightVendorGuid() Axios call error. ex: " + ex);
			LogVerbose("getBitsightVendorGuid() Axios call error. ex.stack: " + ex.stack);
			return "";
		}
	} catch (ex) {
		LogWarn("getBitsightVendorGuid() Error constructing API call. Returning empty string. ex:" + ex);
		LogVerbose("getBitsightVendorGuid() Error constructing API call. Returning empty string. ex.stack:" + ex.stack);
		return "";
	}
}

async function getBitsightRating(sBSGUID) {
	LogVerbose("getBitsightRating() start.");
	//Goal is to make API call to Bitsight VRM to get the bst_entity_guid.
	//API call is based on whether the vendor is managed or monitored
	try {
		if (typeof sBSGUID == "undefined" || sBSGUID == "" || sBSGUID == null) {
			LogInfo("getBitsightRating() sBSGUID is empty. Returning zero.");
			return 0;
		}

		const sURL = params["Bitsight_webroot"] + params["Bitsight_vendorratingpathapi"] + sBSGUID + "/rating"; //https://service.Bitsighttech.com/customer-api/vrm/v1/companies/{bst_entity_guid}/rating

		const sAuth = "Basic " + makeBasicAuth(params["Bitsight_token"]);

		const headers = {
			"Content-Type": "application/json",
			"Accept": "application/json",
			"Authorization": sAuth,
			"X-Bitsight-CONNECTOR-NAME-VERSION": COST_ArcherBitsightVersion,
			"X-Bitsight-CALLING-PLATFORM-VERSION": COST_Platform,
			"X-Bitsight-CUSTOMER": BitsightCustomerName,
		};

		const options = {
			method: "GET",
			url: sURL,
			headers: headers,
			data: {},
		};

		LogVerbose("getBitsightRating() API options=" + JSON.stringify(options));

		try {
			const { data } = await axios.request(options);
			LogVerbose("getBitsightRating() data=" + JSON.stringify(data));

			if (typeof data == "undefined" || data == null) {
				LogError("getBitsightRating() API returned undefined or null.");
				return "";
			} else {
				//Iterate the tags and populate the object

				if (typeof data.ratings == "undefined" || typeof data.ratings[0] == "undefined" || typeof data.ratings[0].rating == "undefined") {
					LogWarn("getBitsightRating() API returned undefined or null for data.ratings[0].rating.");
					return "";
				} else {
					LogVerbose("getBitsightRating() returning data.ratings[0].rating=" + data.ratings[0].rating);
					return data.ratings[0].rating;
				}
			}
		} catch (ex) {
			LogError("getBitsightRating() Axios call error. ex: " + ex);
			LogVerbose("getBitsightRating() Axios call error. ex.stack: " + ex.stack);
			return "";
		}
	} catch (ex) {
		LogWarn("getBitsightRating() Error constructing API call. Returning empty string. ex:" + ex);
		LogVerbose("getBitsightRating() Error constructing API call. Returning empty string. ex.stack:" + ex.stack);
		return "";
	}
}

async function getBitsightRequirementProgress(sBSVRMVendorGUID) {
	LogVerbose("getBitsightRequirementProgress() start.");
	try {
		if (typeof sBSVRMVendorGUID == "undefined" || sBSVRMVendorGUID == "" || sBSVRMVendorGUID == null) {
			LogVerbose("getBitsightRequirementProgress() sBSVRMVendorGUID. Returning 0.");
			return 0;
		}
		//https://service.Bitsighttech.com/customer-api/vrm/v1/vendors/{vendor_guid}/requirements
		//Documentation: https://Bitsight.stoplight.io/docs/VRM-api/51rip3itf833e-vendor-requirements-progress
		const sURL = params["Bitsight_webroot"] + params["Bitsight_vendorrequirementprogresspathapi"] + sBSVRMVendorGUID + "/requirements"; //https://service.Bitsighttech.com/customer-api/vrm/v1/companies/{bst_entity_guid}/rating

		const sAuth = "Basic " + makeBasicAuth(params["Bitsight_token"]);

		const headers = {
			"Content-Type": "application/json",
			"Accept": "application/json",
			"Authorization": sAuth,
			"X-Bitsight-CONNECTOR-NAME-VERSION": COST_ArcherBitsightVersion,
			"X-Bitsight-CALLING-PLATFORM-VERSION": COST_Platform,
			"X-Bitsight-CUSTOMER": BitsightCustomerName,
		};

		const options = {
			method: "GET",
			url: sURL,
			headers: headers,
			data: {},
		};

		LogVerbose("getBitsightRequirementProgress() API options=" + JSON.stringify(options));

		try {
			const { data } = await axios.request(options);
			LogVerbose("getBitsightRequirementProgress() data=" + JSON.stringify(data));

			if (typeof data == "undefined" || data == null) {
				LogError("getBitsightRequirementProgress() API returned undefined or null.");
				return 0;
			} else {
				//Iterate the tags and populate the object

				if (typeof data.progress_average == "undefined" || data.progress_average == null) {
					LogWarn("getBitsightRequirementProgress() API returned undefined or null for data.progress_average.");
					return 0;
				} else {
					LogVerbose("getBitsightRequirementProgress() returning data.progress_average=" + data.progress_average);
					return data.progress_average;
				}
			}
		} catch (ex) {
			LogError("getBitsightRequirementProgress() Axios call error. ex: " + ex);
			LogVerbose("getBitsightRequirementProgress() Axios call error. ex.stack: " + ex.stack);
			return 0;
		}
	} catch (ex) {
		LogWarn("getBitsightRequirementProgress() Error constructing API call. Returning empty string. ex:" + ex);
		LogVerbose("getBitsightRequirementProgress() Error constructing API call. Returning empty string. ex.stack:" + ex.stack);
		return 0;
	}
}

function getArcherContentID(sBSDomain) {
	LogInfo("getArcherContentID() Start. sBSDomain=" + sBSDomain);
	try {
		sBSDomain = sBSDomain.toLowerCase().trim();

		for (let a in aArcherTPPReport) {
			let sArcherDomain = aArcherTPPReport[a].BitsightVRMDomain.toLowerCase().trim();
			if (sArcherDomain == sBSDomain) {
				LogInfo("getArcherContentID() Match found. Returning: " + aArcherTPPReport[a].ArcherContentID);
				return parseInt(aArcherTPPReport[a].ArcherContentID, 10);
			}
		}
	} catch (ex) {
		LogError("getArcherContentID() Error comparing domains. sBSDomain=" + sBSDomain + " ex:" + ex);
		LogVerbose("getArcherContentID() Error comparing domains. sBSDomain=" + sBSDomain + " ex.stack:" + ex.stack);
		return null;
	}

	LogWarn("getArcherContentID() **DID NOT FIND MATCH IN ARCHER AND WILL NOT BE ABLE TO UPDATE. sBSDomain=" + sBSDomain);
	return null;
}

async function UpdateArcherTPPs() {
	//Iterate through the BitsightVRMVendors clean object. For each object with an Archer ID, we will call a function update Archer. Otherwise skip it.

	LogInfo("UpdateArcherTPPs() Start.");
	try {
		for (let b in BitsightVRMVendors) {
			//We only update vendors that we have in Archer.
			if (typeof BitsightVRMVendors[b].iArcherContentID != "undefined" && BitsightVRMVendors[b].iArcherContentID != null && BitsightVRMVendors[b].iArcherContentID > 0) {
				LogInfo("======================================================================================================");
				LogInfo("getArcherContentID() Working on BitsightVRMVendors[b].VRMDomain: " + BitsightVRMVendors[b].VRMDomain);
				BitsightVRMVendors[b].APIStatus = "02-Ready to Update Archer.";
				await UpdateArcherTPP(b);
			}
		}
	} catch (ex) {
		LogError("UpdateArcherTPPs() Error iterating BitsightVRMVendors. ex:" + ex);
		LogVerbose("UpdateArcherTPPs() Error iterating BitsightVRMVendors. ex.stack:" + ex.stack);
	}

	return;
}

async function UpdateArcherTPP(b) {
	//Iterate through the BitsightVRMVendors clean object. For each object with an Archer ID, we will update it. Otherwise skip it.
	let bSuccess = true;

	LogInfo("UpdateArcherTPP() Start.");
	try {
		const LevelID = params["archer_ThirdPartyProfileLevelID"];
		const ContentID = BitsightVRMVendors[b].iArcherContentID;
		//We don't need the domain since it already exists.  BitsightVRMVendors[b].VRMDomain
		const sBSVRMVendorGUID = BitsightVRMVendors[b].VendorGUID;

		LogInfo("************************************************************************/////////////////////////////////////////////////////////////////////////////////////");
		LogInfo("************************************************************************/////////////////////////////////////////////////////////////////////////////////////");
		LogInfo("************************************************************************/////////////////////////////////////////////////////////////////////////////////////");
		LogInfo("************************************************************************/////////////////////////////////////////////////////////////////////////////////////");
		LogInfo("************************************************************************/////////////////////////////////////////////////////////////////////////////////////");
		LogInfo("sBSVRMVendorGUID=" + sBSVRMVendorGUID);
		LogInfo("************************************************************************/////////////////////////////////////////////////////////////////////////////////////");
		LogInfo("************************************************************************/////////////////////////////////////////////////////////////////////////////////////");
		LogInfo("************************************************************************/////////////////////////////////////////////////////////////////////////////////////");
		LogInfo("************************************************************************/////////////////////////////////////////////////////////////////////////////////////");
		LogInfo("************************************************************************/////////////////////////////////////////////////////////////////////////////////////");

		const sBSGUID = BitsightVRMVendors[b].BSGUID;
		//const sBSVRMGUID = BitsightVRMVendors[b].VRMGUID;
		const sBSVRMLifecycle = GetListValuesLifeCycle(BitsightVRMVendors[b].VRMLifecycle); //Need to format for Archer content upload into values list.
		const sBSVRMTags = GetListValuesTags(BitsightVRMVendors[b].VRMTags); //Need to format for Archer content upload into values list.
		const sBSVRMLastUpdatedDate = BitsightVRMVendors[b].VRMLastUpdatedDate;
		const sBSVRMRequirementsCompletion = getInt(BitsightVRMVendors[b].VRMRequirementsCompletion); //Need to ensure it is an integer or Archer will fail.
		const sBSVRMRequirementsDueDate = BitsightVRMVendors[b].VRMRequirementsDueDate;
		const sBSVRMReviewYear = BitsightVRMVendors[b].VRMReviewYear;
		const sBSVRMImpactScore = BitsightVRMVendors[b].VRMImpactScore;
		const sBSVRMTrustScore = BitsightVRMVendors[b].VRMTrustScore;
		const sBSVRMRiskScore = BitsightVRMVendors[b].VRMRiskScore;
		const sBSSecurityRating = BitsightVRMVendors[b].SecurityRating;
		const sBSVRMDeepLink = BitsightVRMVendors[b].VRMDeepLink;

		//Get the Archer Field IDs
		const BSGUID_fid = ArcherTPPFieldParams["Bitsight GUID"].id;
		const BSVRMGUID_fid = ArcherTPPFieldParams["Bitsight VRM GUID"].id;
		const BSVRMLifecycle_fid = ArcherTPPFieldParams["Bitsight VRM Lifecycle"].id;
		const BSVRMTags_fid = ArcherTPPFieldParams["Bitsight VRM Tags"].id;
		const BSVRMLastUpdatedDate_fid = ArcherTPPFieldParams["Bitsight VRM Last Updated Date"].id;
		const BSVRMRequirementsCompletion_fid = ArcherTPPFieldParams["Bitsight VRM Requirements Completion %"].id;
		const BSVRMRequirementsDueDate_fid = ArcherTPPFieldParams["Bitsight VRM Requirements Due Date"].id;
		const BSVRMReviewYear_fid = ArcherTPPFieldParams["Bitsight VRM Review Year"].id;
		const BSVRMImpactScore_fid = ArcherTPPFieldParams["Bitsight VRM Impact Score"].id;
		const BSVRMTrustScore_fid = ArcherTPPFieldParams["Bitsight VRM Trust Score"].id;
		const BSVRMRiskScore_fid = ArcherTPPFieldParams["Bitsight VRM Risk Score"].id;
		const BSSecurityRating_fid = ArcherTPPFieldParams["Bitsight Security Rating"].id;
		const BSVRMDeepLink_fid = ArcherTPPFieldParams["Bitsight VRM Deep Link"].id;

		const postBody = {
			"Content": {
				"LevelId": LevelID,
				"Id": ContentID,
				"Tag": "TPP",
				"FieldContents": {
					[BSGUID_fid]: {
						"Type": ArcherInputTypes["text"],
						"Tag": "Bitsight GUID",
						"Value": sBSGUID,
						"FieldID": BSGUID_fid,
					},
					[BSVRMGUID_fid]: {
						"Type": ArcherInputTypes["text"],
						"Tag": "Bitsight VRM GUID",
						"Value": sBSVRMVendorGUID,
						"FieldID": BSVRMGUID_fid,
					},
					[BSVRMLifecycle_fid]: {
						"Type": ArcherInputTypes["valueslist"],
						"Tag": "Bitsight VRM Lifecycle",
						"Value": sBSVRMLifecycle,
						"Value": {
							"ValuesListIds": sBSVRMLifecycle,
						},
						"FieldID": BSVRMLifecycle_fid,
					},
					[BSVRMTags_fid]: {
						"Type": ArcherInputTypes["valueslist"],
						"Tag": "Bitsight VRM Tags",
						"Value": {
							"ValuesListIds": sBSVRMTags,
						},
						"FieldID": BSVRMTags_fid,
					},
					[BSVRMLastUpdatedDate_fid]: {
						"Type": ArcherInputTypes["date"],
						"Tag": "Bitsight VRM Last Updated Date",
						"Value": sBSVRMLastUpdatedDate,
						"FieldID": BSVRMLastUpdatedDate_fid,
					},
					[BSVRMRequirementsCompletion_fid]: {
						"Type": ArcherInputTypes["numeric"],
						"Tag": "Bitsight VRM Requirements Completion %",
						"Value": sBSVRMRequirementsCompletion,
						"FieldID": BSVRMRequirementsCompletion_fid,
					},
					[BSVRMRequirementsDueDate_fid]: {
						"Type": ArcherInputTypes["date"],
						"Tag": "Bitsight VRM Requirements Due Date",
						"Value": sBSVRMRequirementsDueDate,
						"FieldID": BSVRMRequirementsDueDate_fid,
					},
					[BSVRMReviewYear_fid]: {
						"Type": ArcherInputTypes["numeric"],
						"Tag": "Bitsight VRM Review Year",
						"Value": sBSVRMReviewYear,
						"FieldID": BSVRMReviewYear_fid,
					},
					[BSVRMImpactScore_fid]: {
						"Type": ArcherInputTypes["numeric"],
						"Tag": "Bitsight VRM Impact Score",
						"Value": sBSVRMImpactScore,
						"FieldID": BSVRMImpactScore_fid,
					},
					[BSVRMTrustScore_fid]: {
						"Type": ArcherInputTypes["numeric"],
						"Tag": "Bitsight VRM Trust Score",
						"Value": sBSVRMTrustScore,
						"FieldID": BSVRMTrustScore_fid,
					},
					[BSVRMRiskScore_fid]: {
						"Type": ArcherInputTypes["numeric"],
						"Tag": "Bitsight VRM Risk Score",
						"Value": sBSVRMRiskScore,
						"FieldID": BSVRMRiskScore_fid,
					},
					[BSSecurityRating_fid]: {
						"Type": ArcherInputTypes["numeric"],
						"Tag": "Bitsight Security Rating",
						"Value": sBSSecurityRating,
						"FieldID": BSSecurityRating_fid,
					},
					[BSVRMDeepLink_fid]: {
						"Type": ArcherInputTypes["text"],
						"Tag": "Bitsight VRM Deep Link",
						"Value": sBSVRMDeepLink,
						"FieldID": BSVRMDeepLink_fid,
					},
				},
			},
		}; //End of postBody

		LogVerbose("UpdateArcherTPP() postbody=" + JSON.stringify(postBody));

		try {
			let data = null;

			const sUrl = params["archer_webroot"] + params["archer_rest_root"] + params["archer_contentpath"];

			const httpConfig = {
				method: "PUT",
				url: sUrl,
				headers: {
					"Authorization": 'Archer session-id="' + sArcherSessionToken + '"',
					"Content-Type": "application/json",
					Accept: "application/json,text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8",
				},
				data: postBody,
			};

			LogVerbose("UpdateArcherTPP() API call httpConfig=" + JSON.stringify(httpConfig));

			await axios(httpConfig)
				.then(function (res) {
					data = res.data;
					LogVerbose("UpdateArcherTPP() Axios call complete. data=" + JSON.stringify(data));
				})
				.catch(function (error) {
					bSuccess = false;
					totalReportErrors++;
					BitsightVRMVendors[b].APIStatus = "03a-Archer TPP Updated FAILED.";
					LogError("UpdateArcherTPP() Axios call error. Err: " + error);
					LogVerbose("UpdateArcherTPP() Axios call error. Err.stack: " + error.stack);
				});

			if (
				typeof data != "undefined" &&
				typeof data.RequestedObject != "undefined" &&
				typeof data.RequestedObject.Id != "undefined" &&
				typeof data.IsSuccessful != "undefined" &&
				data.IsSuccessful == true
			) {
				LogVerbose("UpdateArcherTPP() Archer TPP update successful. ID=" + data.RequestedObject.Id);
				totalReportSuccess++;
				BitsightVRMVendors[b].APIStatus = "03-Archer TPP Updated Successfully.";
			} else {
				bSuccess = false;
				totalReportErrors++;
				BitsightVRMVendors[b].APIStatus = "03b-Archer TPP Updated FAILED.";
				LogError("UpdateArcherTPP() ERROR updating TPP. API result not sucessful.");
			}
		} catch (ex) {
			bSuccess = false;
			totalReportErrors++;
			BitsightVRMVendors[b].APIStatus = "03c-Archer TPP Updated FAILED.";
			LogError("UpdateArcherTPP() Error constructing call or parsing result. ex: " + ex);
			LogVerbose("UpdateArcherTPP() Error constructing call or parsing result. ex.stack: " + ex.stack);
		}
	} catch (ex) {
		bSuccess = false;
		totalReportErrors++;
		BitsightVRMVendors[b].APIStatus = "03d-Archer TPP Updated FAILED.";
		LogError("UpdateArcherTPP() Error creating Archer postbody for " + BitsightVRMVendors[b].VRMDomain + " ex:" + ex);
		LogVerbose("UpdateArcherTPP() Error creating Archer postbody for " + BitsightVRMVendors[b].VRMDomain + " ex.stack:" + ex.stack);
	}
	return bSuccess;
}

function GetListValuesLifeCycle(LifeCycle) {
	LogInfo("GetRatingListValues() Start.");
	//Function to identify and return an array of values list IDs.
	//Example Input for the LifeCycle:
	// 	{
	// 		"Bitsight_life_cycle_stage_guid": "1d860d16-b9d6-4c63-bb08-abf4368cf4c9",
	// 		"Name": "Base Pending",
	// 		"ArchertaglistID": 81311
	// 	}
	//Desired Output: [81311]

	try {
		LogVerbose("GetListValuesLifeCycle() LifeCycle=" + JSON.stringify(LifeCycle));

		if (typeof LifeCycle == "undefined" || LifeCycle == null || typeof LifeCycle.ArcherLifecyclestagelistID == "undefined") {
			LogInfo("GetListValuesLifeCycle() LifeCycle empty.");
			return [];
		}

		let tmp = [];
		tmp[tmp.length] = LifeCycle.ArcherLifecyclestagelistID;

		return tmp;
	} catch (ex) {
		LogInfo("GetListValuesLifeCycle() Error iterating LifeCycle - returning [].");
		return [];
	}
}

function GetListValuesTags(Tags) {
	LogInfo("GetListValuesTags() Start.");

	//Function to iterate and return an array of Tag values list IDs.
	//Example Input:
	// [
	// 	{
	// 		"Bitsight_tag_guid": "00eaa285-ba77-4ac5-bc96-e0793c7f6db5",
	// 		"Name": "DougTestTag",
	// 		"ArchertaglistID": 81303
	// 	},
	// 	{
	// 		"Bitsight_tag_guid": "201e5420-9723-4c4d-8922-97ff2dccc494",
	// 		"Name": "DougTestTag2",
	// 		"ArchertaglistID": 81304
	// 	}
	// ],
	//Desired Output: [81303,81304]

	try {
		LogVerbose("GetListValuesLifeCycle() Tags=" + JSON.stringify(Tags));
		if (typeof Tags == "undefined" || Tags == null || Tags.length == 0) {
			LogInfo("GetListValuesTags() Tags empty.");
			return [];
		}

		let tmp = [];
		for (var t in Tags) {
			tmp[tmp.length] = Tags[t].ArchertaglistID;
		}

		return tmp;
	} catch (ex) {
		LogInfo("GetListValuesTags() Error iterating Tags - returning [].");
		return [];
	}
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////END CORE FUNCTIONALITY
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

//Main driver function of the overall process
async function main() {
	//flag of successful transactions just for inside this loop.
	let bSuccess = true;
	LogInfo("main() Launching...");

	//Login to Archer
	bSuccess = await ArcherLogin();

	//If we successfully received reports continue
	if (bSuccess === true) {
		LogInfo("main() - Got Archer Session token.");
		//Get Archer Version for Bitsight Stats - Not critical and if it fails, will not fail the entire process
		await GetArcherVersion();

		//Get the Archer App ID and Fields - Needed to construct search criteria and update the Archer record after Adding to Bitsight.
		bSuccess = await getArcherTPPFieldIDs();

		if (bSuccess === true) {
			bSuccess = await getArcherTPPsWithDomain();

			if (bSuccess === true) {
				LogInfo("getArcherTPPsWithDomain successful, make requests to Bitsight to get tag and lifecycle list values.");

				bSuccess = await getBitsightListValues();

				if (bSuccess === true) {
					LogInfo("getBitsightListValues successful, ensure they match Archer or update Archer.");
					bSuccess = await ValidateAndUpdateListValues();

					if (bSuccess === true) {
						bSuccess = await BitsightListAllVendors();

						if (bSuccess === true) {
							await UpdateArcherTPPs(); //Note individual vendors will have success/fail tracked
						}
					}
				}
			}
		}
	} else if (bSuccess === false) {
		LogInfo("main() - Error obtaining Archer Session Token. Failing overall.");
	}
}

//This function runs the overall process.
async function runAwait() {
	try {
		//Launch main process/functionality
		await main();
	} catch (ex) {
		LogInfo("==================================================================================");
		LogError("runAwait() Error Executing main(). Err: " + ex);
		LogError("runAwait() Error Executing main(). Err.stack: " + ex.stack);
		LogInfo("==================================================================================");
	} finally {
		//Wrap up and show final Archer Original Requested object which has the status:
		LogInfo("==================================================================================");
		LogInfo("runAwait() Final aArcherTPPReport: " + JSON.stringify(aArcherTPPReport));
		LogInfo("==================================================================================");

		//Wrap up and show final object Bitsight object which has the status:
		LogInfo("==================================================================================");
		LogInfo("runAwait() Final BitsightVRMVendors: " + JSON.stringify(BitsightVRMVendors));
		LogInfo("==================================================================================");

		//Show final stats
		LogInfo("==================================STATS===========================================");
		LogInfo("totalReportRequests: " + totalReportRequests.toString());
		LogInfo(" totalReportSuccess: " + totalReportSuccess.toString());
		LogInfo("  totalReportErrors: " + totalReportErrors.toString());
		LogInfo("        totalErrors: " + totalErrors.toString());
		LogInfo("      totalWarnings: " + totalWarnings.toString());

		//Build summary that gets data into the API Monitoring
		let sSummary = JSON.stringify({
			"totalReportRequests": totalReportRequests,
			"totalReportSuccess": totalReportSuccess,
			"totalReportErrors": totalReportErrors,
			"#Errors": totalErrors,
			"#Warnings": totalWarnings,
		});

		LogInfo("==================================================================================");
		//Sanity check for matches. We should have the same numbers for found and added.
		if (bOverallSuccessful === true && totalReportRequests == totalReportSuccess) {
			LogInfo("runAwait() No Errors found and MATCH! Total found equals updated!");
			//We could consider this a success if they match, but could be a false sense of accomplishment.
			//Since we set bOverallSuccessful if a LogError() is called, honor that.
			bOverallSuccessful = true;
		} else {
			LogInfo("runAwait() ERROR or MISMATCH! Total found does NOT equal updated!");
			bOverallSuccessful = false;
		}

		LogInfo("==================================================================================");
		LogInfo("==================================================================================");
		LogInfo("runAwait() Summary of Errors/Warnings from debugLog:\r\n" + sErrorLogDetails);
		LogInfo("==================================================================================");
		LogInfo("==================================================================================");

		if (bOverallSuccessful == true) {
			LogSaaSSuccess();
			LogInfo("runAwait() Finished SUCCESSFULLY");
		} else {
			try {
				LogSaaSError();
			} catch (ex) {
				LogInfo("runAwait() ERROR LogSaaSError. SaaSErr02: " + ex);
				LogInfo("runAwait() ERROR LogSaaSError. SaaSErr02: stack=" + ex.stack);
			}

			LogInfo("runAwait() Finished with ERRORS");
		}
	}
}

//start async app driver for overall functionality and then determines final success/fail resolution.
runAwait();
